
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(42)  # Set a seed for reproducibility

from scipy.stats import gaussian_kde

# Generate data for the plot
x = np.linspace(0, 11, 1000)  # Time in months
# Simulated performance data for seven different football teams
y = [
    np.abs(np.random.uniform(2, 5) * np.sin(0.5 * x * (index + 1))
           + np.random.normal(5, 1, len(x)))  # Ensure positive values for goals scored
    for index in range(7)
]

# Combine all team data into a single array for density estimation
all_data = np.concatenate(y)

# Extracted variables
xlabel_text = "Months"
ylabel_text = "Goals Scored"
title_text = "Density of Monthly Goals Scored by Football Teams"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 6))

# Perform Kernel Density Estimation (KDE) using scipy
kde = gaussian_kde(all_data, bw_method='scott')
x_dens = np.linspace(min(all_data), max(all_data), 1000)
density = kde(x_dens)

# Plot the density curve
ax.plot(x_dens, density, color='purple', lw=2)
ax.fill_between(x_dens, density, color='purple', alpha=0.3)  # Fill under the curve

# Set labels and title
ax.set_xlabel(xlabel_text)
ax.set_ylabel(ylabel_text)
ax.set_title(title_text)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.spines["bottom"].set_visible(True)
ax.spines["left"].set_visible(True)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)

# Add grid for readability
plt.grid(True, linestyle=":", alpha=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Display the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_39.pdf", bbox_inches="tight")
