# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)  # Set a seed for reproducibility

from scipy.stats import gaussian_kde

# Generate data for the plot
x = np.linspace(0, 24, 1000)  # Time in months, over 2 years
# Simulated data for different player types with realistic sports trends
y_data = [
    np.clip(
        np.abs(
            np.random.uniform(20, 30) * np.cos(0.2 * x * (index + 1)) * (1 - 0.01 * x)
            + np.random.normal(0, 1, x.shape)
        ),
        0,
        50,
    )  # Clip to avoid unrealistic values
    for index in range(3)
]

# Combine all player data into a single array for density estimation
all_data = np.concatenate(y_data)

# Extracted variables
xlabel_text = "Months"
ylabel_text = "Density"
title_text = "Density of Player Scoring Efficiency Over Time"
legend_title = "Player Types"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 7))

# Colors for different density curves
colors = plt.cm.viridis(np.linspace(0, 1, len(y_data)))

# Plot separate density curves for each player type
for i, y in enumerate(y_data):
    kde = gaussian_kde(y, bw_method="scott")
    x_dens = np.linspace(min(y), max(y), 1000)
    density = kde(x_dens)
    ax.plot(x_dens, density, color=colors[i], lw=2, label=f"Player Type {i+1}")
    ax.fill_between(x_dens, density, color=colors[i], alpha=0.2)

# Set labels and title
ax.set_xlabel(xlabel_text)
ax.set_ylabel(ylabel_text)
ax.set_title(title_text)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.spines["bottom"].set_visible(True)
ax.spines["left"].set_visible(True)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)

# Add legend
plt.legend(title=legend_title, loc="upper right", fontsize=10)

# Add grid for readability
plt.grid(True, linestyle=":", alpha=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Display the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_40.pdf", bbox_inches="tight")
