
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Fix a seed for reproducibility
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(42)
# Generate bimodal distribution for vehicle speeds data
# Simulate vehicle speeds for two road types (Urban Roads and Highways)
speeds_urban = np.random.normal(loc=40, scale=10, size=400)  # Urban roads speeds
speeds_highway = np.random.normal(loc=90, scale=15, size=400)  # Highway speeds
vehicle_speeds_data = np.concatenate([speeds_urban, speeds_highway])

# X-axis values for KDE
xs = np.linspace(0, 150, 200)

# Axes Limits and Labels
title = "KDE Plot of Vehicle Speeds on Urban Roads and Highways"
xlabel_value = "Vehicle Speed (km/h)"
ylabel_value = "Density"
legend_label = "Speed Density for Urban Roads and Highways"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
xticks_values = [0, 20, 40, 60, 80, 100, 120, 140, 160]
xticklabels = ["0", "20", "40", "60", "80", "100", "120", "140", "160"]
yticks_values = [0, 0.005, 0.01, 0.015, 0.02]
yticklabels = ["0", "0.005", "0.01", "0.015", "0.02"]
xlim_values = [0, 150]
ylim_values = [0, 0.02]
title_fontsize = 16
title_weight = "bold"
label_fontsize = 14
legend_loc = "upper right"
legend_fontsize = 14

# Set the figure size
fig, ax = plt.subplots(figsize=(10, 6))

# Create the KDE plot with adjusted x-axis range
density = gaussian_kde(vehicle_speeds_data)
density.covariance_factor = lambda: 0.25
density._compute_covariance()
plt.fill_between(xs, density(xs), color="#99c2ff", edgecolor="navy", alpha=0.7)

# Customize ticks and labels
ax.set_xticks(xticks_values)
ax.set_xticklabels(xticklabels, fontsize=label_fontsize)

ax.set_yticks(yticks_values)
ax.set_yticklabels(yticklabels, fontsize=label_fontsize)

plt.xlim(xlim_values)
plt.ylim(ylim_values)

# Set the title and labels
plt.title(title, fontsize=title_fontsize, weight=title_weight)
plt.xlabel(xlabel_value, fontsize=label_fontsize)
plt.ylabel(ylabel_value, fontsize=label_fontsize)

# Add a legend
ax.legend([legend_label], loc=legend_loc, fontsize=legend_fontsize)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_60.pdf", bbox_inches="tight")
