

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
# Fix a seed for reproducibility
import numpy as np

np.random.seed(42)

# Generate synthetic data for temperature change in two cities
# City A: Minor changes
temp_change_city_a = np.random.normal(loc=0, scale=2, size=400)
# City B: Major changes
temp_change_city_b = np.random.normal(loc=5, scale=4, size=400)

# Combine the data
temp_change_data = np.concatenate([temp_change_city_a, temp_change_city_b])

# X-axis values for KDE
xs = np.linspace(-10, 20, 200)

# Axes Limits and Labels
title = "KDE Plot of Temperature Change Distribution for Two Cities"
xlabel_value = "Temperature Change (°C)"
ylabel_value = "Density"
legend_label = "Density of Temperature Change for Cities A and B"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
xticks_values = [-10, 0, 10, 20]
xticklabels = ["-10", "0", "10", "20"]
yticks_values = [0, 0.05, 0.1, 0.15]
yticklabels = ["0", "0.05", "0.1", "0.15"]
xlim_values = [-10, 20]
ylim_values = [0, 0.16]
title_fontsize = 14
title_weight = "bold"
label_fontsize = 12
legend_loc = "upper right"
legend_fontsize = 12

# Set the figure size
fig, ax = plt.subplots(figsize=(10, 6))

# Create the KDE plot with adjusted x-axis range
density = gaussian_kde(temp_change_data)
density.covariance_factor = lambda: 0.3
density._compute_covariance()
plt.fill_between(xs, density(xs), color="#93c572", edgecolor="#6b8e23", alpha=0.6)

# Customize ticks and labels
ax.set_xticks(xticks_values)
ax.set_xticklabels(xticklabels, fontsize=label_fontsize)

ax.set_yticks(yticks_values)
ax.set_yticklabels(yticklabels, fontsize=label_fontsize)

plt.xlim(xlim_values)
plt.ylim(ylim_values)

# Set the title and labels
plt.title(title, fontsize=title_fontsize, weight=title_weight)
plt.xlabel(xlabel_value, fontsize=label_fontsize)
plt.ylabel(ylabel_value, fontsize=label_fontsize)

# Add a legend
ax.legend([legend_label], loc=legend_loc, fontsize=legend_fontsize)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_61.pdf", bbox_inches="tight")
