

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
from scipy.stats import gaussian_kde

# Set random seed for reproducibility
np.random.seed(101)

# Generate a societal domain dataset simulating age distribution
children = np.random.normal(loc=7, scale=3, size=500)
working_age = np.random.normal(loc=40, scale=10, size=1500)
seniors = np.random.normal(loc=70, scale=7, size=300)
age_data = np.concatenate([children, working_age, seniors])
xs = np.linspace(0, 100, 300)

# Axes Limits and Labels
title = "KDE Plot of Simulated Age Distribution"
xlabel_value = "Age (in years)"
ylabel_value = "Density"

xticks_values = [0, 20, 40, 60, 80, 100]
xticklabels = ["0", "20", "40", "60", "80", "100"]
yticks_values = [0, 0.01, 0.02, 0.03, 0.04]
yticklabels = ["0", "0.01", "0.02", "0.03", "0.04"]
xlim_values = [0, 100]
ylim_values = [0, 0.05]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

# Set the figure size
fig, ax = plt.subplots(figsize=(12, 6))

# Create the KDE plot with adjusted x-axis range
density = gaussian_kde(age_data)
density.covariance_factor = lambda: 0.25
density._compute_covariance()
plt.fill_between(xs, density(xs), color="#add8e6", edgecolor="#4682b4")

ax.set_xticks(xticks_values)
ax.set_xticklabels(xticklabels)

ax.set_yticks(yticks_values)
ax.set_yticklabels(yticklabels)

plt.xlim(xlim_values)
plt.ylim(ylim_values)
# Set the title and labels
plt.title(title, fontsize=14)
plt.xlabel(xlabel_value, fontsize=12)
plt.ylabel(ylabel_value, fontsize=12)

# Add grid for better readability
ax.grid(True, which='both', linestyle='--', linewidth=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_64.pdf", bbox_inches="tight")

