

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generate sample population weekly exercise hours for three different age groups
young_adults_exercise = np.random.normal(loc=5, scale=2, size=1000)
middle_aged_exercise = np.random.normal(loc=3, scale=1.5, size=1000)
seniors_exercise = np.random.normal(loc=2, scale=1, size=1000)

labels = ["Young Adults Exercise Hours", "Middle-aged Adults Exercise Hours", "Seniors Exercise Hours"]
avxlabel = "Average Exercise Hours (Young Adults)"
xlabel = "Weekly Exercise Hours"
ylabel = "Density"
title = "Weekly Exercise Hours Density for Different Age Groups"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
grid_linestyle = "-."
grid_linewidth = 0.7
grid_alpha = 0.8
line_linestyle = "-"
line_linewidth = 2.5
fill_alpha = 0.3

fig, ax = plt.subplots(figsize=(10, 6))

# Define new colors
colors = ["#3498db", "#e74c3c", "#2ecc71"]

# Plot the density plots
for data, color, label in zip(
    [young_adults_exercise, middle_aged_exercise, seniors_exercise],
    colors,
    labels,
):
    density = gaussian_kde(data)
    xs = np.linspace(min(data) - 5, max(data) + 5, 200)
    density.covariance_factor = lambda: 0.35  # Adjust for smoother curves
    density._compute_covariance()
    plt.plot(
        xs,
        density(xs),
        color=color,
        label=label,
        linestyle=line_linestyle,  # solid line
        linewidth=line_linewidth,  # slightly thicker lines
    )
    plt.fill_between(
        xs, density(xs), color=color, alpha=fill_alpha
    )  # strong fill for shade

# Plot a vertical line indicating the average exercise hours for young adults
plt.axvline(
    x=np.mean(young_adults_exercise),
    color="#9b59b6",
    linestyle=line_linestyle,
    linewidth=line_linewidth,
    label=avxlabel,
)

# Set labels and title
ax.set_xlim(
    min(min(young_adults_exercise), min(middle_aged_exercise), min(seniors_exercise)) - 5,
    max(max(young_adults_exercise), max(middle_aged_exercise), max(seniors_exercise)) + 5,
)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# Show grid with improved linestyle
plt.grid(True, linestyle=grid_linestyle, linewidth=grid_linewidth, alpha=grid_alpha)

# Add legend
plt.legend()

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("density_66.pdf", bbox_inches="tight")
