
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generate sample car sales volume data
spring_sales = np.random.normal(loc=50, scale=10, size=1000)  # in thousands
summer_sales = np.random.normal(loc=100, scale=20, size=1000)  # in thousands
winter_sales = np.random.normal(loc=30, scale=5, size=1000)  # in thousands

labels = ["Spring Car Sales", "Summer Car Sales", "Winter Car Sales"]
avxlabel = "Average Spring Car Sales"
xlabel = "Car Sales Volume (in thousands)"
ylabel = "Density"
title = "Seasonal Car Sales Volume Density Distributions"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
grid_linestyle = "-"
grid_linewidth = 0.5
grid_alpha = 0.7
line_linestyle = "--"
line_linewidth = 2
fill_alpha = 0.25

fig, ax = plt.subplots(figsize=(10, 6))

# Plot the density plots
for data, color, label in zip(
    [spring_sales, summer_sales, winter_sales],
    ["#77DD77", "#AEC6CF", "#CFCFC4"],  # Specific colors for Spring, Summer, and Winter
    labels,
):
    density = gaussian_kde(data)
    xs = np.linspace(min(data) - 10, max(data) + 10, 200)
    density.covariance_factor = lambda: 0.25  # Adjust for smoother curves
    density._compute_covariance()
    plt.plot(
        xs,
        density(xs),
        color=color,
        label=label,
        linestyle=line_linestyle,  # dashed line
        linewidth=line_linewidth,  # thicker lines
    )
    plt.fill_between(
        xs, density(xs), color=color, alpha=fill_alpha
    )  # fill for shade

# Plot a vertical line indicating the average spring car sales
plt.axvline(
    x=np.mean(spring_sales),
    color="red",
    linestyle=line_linestyle,
    linewidth=line_linewidth,
    label=avxlabel,
)

# Set labels and title
ax.set_xlim(
    min(min(spring_sales), min(summer_sales), min(winter_sales)) - 10,
    max(max(spring_sales), max(summer_sales), max(winter_sales)) + 10,
)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# Show grid with a subtle linestyle
plt.grid(True, linestyle=grid_linestyle, linewidth=grid_linewidth, alpha=grid_alpha)

# Add legend
plt.legend()

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("density_68.pdf", bbox_inches="tight")
