

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
from scipy.stats import gaussian_kde

# Generate synthetic movie rating data for different genres
np.random.seed(123)

# Action movies, average ratings around 7.5, moderate variance
ratings_action = np.random.normal(loc=7.5, scale=1.0, size=1000)

# Romance movies, average ratings around 6.5, moderate variance
ratings_romance = np.random.normal(loc=6.5, scale=1.0, size=1000)

# Documentaries, average ratings around 8.2, lower variance
ratings_documentary = np.random.normal(loc=8.2, scale=0.8, size=1000)

# Labels for the new data
labels = ["Action Movies", "Romance Movies", "Documentaries"]
avxlabel = "Overall Average Rating"
xlabel = "Movie Ratings (out of 10)"
ylabel = "Density"
title = "Distribution of Movie Ratings by Genre"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the figure and axis
fig, ax = plt.subplots(figsize=(9, 6))

# Define new colors for the different genres
colors = ["#1f77b4", "#2ca02c", "#ff7f0e"]  # blue, green, orange

# Plot the density plots
for data, color, label in zip([ratings_action, ratings_romance, ratings_documentary], colors, labels):
    density = gaussian_kde(data)
    xs = np.linspace(2, 10, 200)
    density.covariance_factor = lambda: 0.4
    density._compute_covariance()
    plt.fill_between(xs, density(xs), color=color, alpha=0.3, label=label)

# Plot the overall average rating line
overall_avg_rating = np.mean([ratings_action.mean(), ratings_romance.mean(), ratings_documentary.mean()])
plt.axvline(x=overall_avg_rating, color="purple", linestyle="--", label=avxlabel)

# Set chart labels and title
ax.set_xlim(2, 10)
ax.set_xticks(np.arange(2, 11, 1))
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
plt.title(title)

# Show grid
plt.grid(True, linestyle="--", alpha=0.6)

# Add legend
plt.legend()

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("density_71.pdf", bbox_inches="tight")

