
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Data for the bar charts
news_sources = ["Social Media", "TV", "Newspapers", "Online News Sites"]
avg_time_spent = [9.2, 7.5, 3.4, 6.8]  # Average time spent in hours per week
trust_rate = [40, 65, 60, 55]  # Trust rate in percentage
error_time_spent = [0.7, 0.5, 0.4, 0.6]  # Error values for time spent
error_trust_rate = [5, 4, 3, 4]  # Error values for trust rate
x = np.arange(len(avg_time_spent))  # x-coordinates for the bars
labels = ["Average Time Spent (hrs/week)", "Trust Rate (%)"]
title = "Comparison of Time Spent and Trust Rate Across News Sources"
ylims = [[0, 11], [30, 70]]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax1 = plt.subplots(figsize=(10, 6))
# Bar width
width = 0.35

# Colors
color_avg_time_spent = "#8da0cb"
color_trust_rate = "#66c2a5"

# Plotting 'avg_time_spent' on the primary y-axis
bars1 = ax1.bar(
    x - width / 2,
    avg_time_spent,
    width,
    label=labels[0],
    color=color_avg_time_spent,
    yerr=error_time_spent,
    capsize=5,
    edgecolor="black",
)

# Create the secondary y-axis for 'trust_rate'
ax2 = ax1.twinx()
bars2 = ax2.bar(
    x + width / 2,
    trust_rate,
    width,
    label=labels[1],
    color=color_trust_rate,
    yerr=error_trust_rate,
    capsize=5,
    edgecolor="black",
)

# Adding annotations directly on the bars for clarity
for i, bar in enumerate(bars1):
    height = bar.get_height()
    label_x_pos = bar.get_x() + bar.get_width() / 2
    ax1.text(
        label_x_pos,
        height - error_time_spent[i] - 0.3,
        f"{height} hrs",
        rotation=90,
        ha="center",
        va="bottom" if height < 0 else "top",
    )
for j, bar in enumerate(bars2):
    height = bar.get_height()
    label_x_pos = bar.get_x() + bar.get_width() / 2
    ax2.text(
        label_x_pos,
        height - error_trust_rate[j] - 1,
        f"{height}%",
        rotation=90,
        ha="center",
        va="bottom" if height < 0 else "top",
    )


fig.suptitle(title)
# Adding labels, title, and custom x-axis tick labels
ax1.set_ylabel(labels[0], color=color_avg_time_spent)
ax2.set_ylabel(labels[1], color=color_trust_rate)
ax1.set_xticks(x)
ax1.set_xticklabels(news_sources)

# Add a horizontal line at y=0 if needed
ax1.axhline(0, color="grey", linewidth=0.8)

# Adjusting y-axis limits to fit the annotations and errors
ax1.set_ylim(ylims[0])
ax2.set_ylim(ylims[1])

# Adding grid lines for better readability
ax1.yaxis.grid(linestyle="--", linewidth="0.5", color="grey")
ax1.set_axisbelow(True)

ax2.yaxis.grid(linestyle="--", color="grey")
ax2.set_axisbelow(True)
# Adding legend
fig.legend(loc="upper center", bbox_to_anchor=(0.5, 1.05), ncol=2)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout
plt.tight_layout()
plt.savefig("errorbar_105.pdf", bbox_inches="tight")
