# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.colors as mcolors

# ===================
# Part 2: Data Preparation
# ===================
# Data for different economic indicators
categories = [
    "Region A",
    "Region B",
    "Region C",
    "Region D",
    "Region E",
    "Region F",
    "Region G",
]
gdp_growth = [
    3.5,
    4.2,
    2.8,
    5.0,
    3.0,
    4.8,
    2.5,
]  # GDP growth percentage
inflation_rate = [
    -1.5,
    -2.2,
    -1.8,
    -3.0,
    -1.0,
    -2.5,
    -1.2,
]  # Negative values for inflation rate

# Updated label variables
xlabel = "Regions"
ylabel = "Percentage"
title = "Economic Indicators by Region"
legendlabels = ["Inflation Rate", "GDP Growth"]
textlabels = ["Inflation", "GDP"]

# Error data for each category
gdp_errors = [0.5, 0.4, 0.6, 0.3, 0.4, 0.2, 0.3]
inflation_errors = [0.2, 0.3, 0.2, 0.1, 0.3, 0.1, 0.1]

suptitle = "Economic Indicators"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axis
fig, ax = plt.subplots(figsize=(10, 6))  # Adjusted size for clarity

# Define colors
gdp_colors = [mcolors.TABLEAU_COLORS[color] for color in mcolors.TABLEAU_COLORS]
inflation_colors = [
    mcolors.to_rgba(color, alpha=0.6) for color in gdp_colors
]  # Slightly lighter for inflation

# Plot vertical bar charts with error bars
ax.bar(
    categories,
    inflation_rate,
    color=inflation_colors,
    edgecolor="black",
    yerr=inflation_errors,
    label=legendlabels[0],
    capsize=3,
)
ax.bar(
    categories,
    gdp_growth,
    bottom=0,
    color=gdp_colors,
    edgecolor="black",
    yerr=gdp_errors,
    label=legendlabels[1],
    capsize=3,
)

# Adding labels within bars for values
for i, (inf_val, gdp_val) in enumerate(zip(inflation_rate, gdp_growth)):
    ax.text(i, inf_val / 2, f"{abs(inf_val)}%", va="center", ha="center", color="gray")
    ax.text(i, gdp_val / 2, f"{gdp_val}%", va="center", ha="center", color="gray")

# Customizing axis and layout
ax.tick_params(axis="x", which="both", bottom=False)  # Remove x-axis tick marks
ax.set_ylim(min(inflation_rate) - 1, max(gdp_growth) + 1)
ax.set_ylabel(ylabel)
ax.set_xlabel(xlabel)
ax.set_title(title)
ax.set_yticks([])
ax.spines["left"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.spines["top"].set_visible(False)
ax.axhline(0, color="black")  # Draw a horizontal line at y=0 for separation

# Super Title
fig.suptitle(suptitle, fontsize=16)

# Labels at the ends
ax.text(-0.5, -3, textlabels[0], ha="right", va="bottom", rotation="vertical")
ax.text(-0.5, 3, textlabels[1], ha="right", va="bottom", rotation="vertical")

# Add legend
ax.legend(legendlabels, loc="upper right")

# ===================
# Part 4: Saving Output
# ===================
# Show plot
plt.tight_layout()
plt.savefig("errorbar_109.pdf", bbox_inches="tight")
