
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Data setup
import numpy as np

np.random.seed(0)

# Regions and X-axis points
regions = ["Region A", "Region B", "Region C", "Region D"]
x = np.arange(len(regions))

# Sales data for two product categories in different regions
# These are fabricated values for demonstration
product1_sales = np.array([25000, 27000, 24000, 26000])  # Product 1 sales
product2_sales = np.array([45000, 47000, 42000, 44000])  # Product 2 sales

# Errors for both metrics
product1_sales_errors = np.array([2000, 2500, 2200, 2100])
product2_sales_errors = np.array([3500, 3000, 3300, 3200])

# Labels for the data series
labels = ["Product 1 Sales", "Product 2 Sales"]
ylabels = ["Product 1 Sales ($)", "Product 2 Sales ($)"]

# Other text labels and titles
title = "Sales Data Across Regions"
ylims = [[20000, 30000], [35000, 50000]]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting setup
fig, ax1 = plt.subplots(figsize=(8, 5))

# Bar width and hatch patterns
width = 0.35
hatch_patterns = ["//", "\\"]

# Colors for the bars
colors = ["#4CAF50", "#2196F3"]  # Green and blue shades

# Plot data on the left y-axis (Product 1 sales)
ax1.bar(
    x,
    product1_sales,
    width,
    color=colors[0],
    hatch=hatch_patterns[0],
    label=labels[0],
    yerr=product1_sales_errors,
    capsize=5,
    edgecolor="black",
)

# Create a second y-axis sharing the same x-axis (Product 2 sales)
ax2 = ax1.twinx()
ax2.bar(
    x + width,
    product2_sales,
    width,
    color=colors[1],
    hatch=hatch_patterns[1],
    label=labels[1],
    yerr=product2_sales_errors,
    capsize=5,
    edgecolor="black",
)

# Set the x-ticks to be in the middle of the two bars and add the labels
ax1.set_xticks(x + width / 2)
ax1.set_xticklabels(regions)

# Add a legend
ax1.legend(loc="upper left")
ax2.legend(loc="upper right")

# Set labels for y-axes
ax1.set_ylabel(ylabels[0], color=colors[0])
ax2.set_ylabel(ylabels[1], color=colors[1])

# Set colors for y-axes
ax1.tick_params(axis="y", colors=colors[0])
ax2.tick_params(axis="y", colors=colors[1])

# Set the limits for y-axes to fit the data
ax1.set_ylim(ylims[0])
ax2.set_ylim(ylims[1])

# Set title for the chart
plt.title(title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("errorbar_119.pdf", bbox_inches="tight")
