

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Data
user_groups = ["Tech Savvy", "Business Users", "Casual Users", "Remote Workers"]
tools = ["Zoom", "Microsoft Teams", "Slack", "Google Meet", "Discord", "Skype"]
effectiveness_means = np.array(
    [
        [85, 75, 70, 80, 65, 60],  # Tech Savvy
        [80, 70, 65, 75, 60, 55],  # Business Users
        [70, 60, 55, 65, 50, 45],  # Casual Users
        [90, 80, 75, 85, 70, 65],  # Remote Workers
    ]
)
effectiveness_std = np.array(
    [
        [5, 6, 5, 7, 6, 5],  # Tech Savvy
        [6, 5, 6, 5, 6, 7],  # Business Users
        [7, 6, 5, 5, 7, 6],  # Casual Users
        [5, 7, 6, 5, 7, 6],  # Remote Workers
    ]
)
legendtitle = "Remote Communication Tools"
xlabel = "User Groups"
ylabel = "Effectiveness (%)"
ylim = [0, 100]

# Labels for plot
title = "Effectiveness of Different Remote Communication Tools"
subtitle = "Evaluated by Different User Groups"


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set figure size to match the original image's dimensions
fig, ax = plt.subplots(figsize=(12, 7))

# colors
colors = [
    "#ff6f61",  # Zoom
    "#6b5b95",  # Microsoft Teams
    "#88b04b",  # Slack
    "#f7cac9",  # Google Meet
    "#92a8d1",  # Discord
    "#034f84",  # Skype
]

# Bar width
bar_width = 0.15
bar_width_main = 0.75

# Set position of bar on X axis
r = np.arange(len(user_groups))

# Draw bars for 'Zoom' tool
i = tools.index("Zoom")
ax.bar(
    r + (i + 3) * bar_width,
    effectiveness_means[:, i],
    yerr=effectiveness_std[:, i],
    width=bar_width_main,
    label=tools[i],
    capsize=5,
    color=colors[i],
    hatch="//",
    edgecolor="black",
)

# Draw bars for other tools
for i in range(len(tools)):
    if tools[i] == "Zoom":
        continue
    ax.bar(
        r + i * bar_width,
        effectiveness_means[:, i],
        yerr=effectiveness_std[:, i],
        width=bar_width,
        label=tools[i],
        capsize=5,
        color=colors[i],
        edgecolor="black",
    )

# Add xticks on the middle of the group bars
ax.set_xlabel(xlabel)
ax.set_xticks(r + bar_width * (len(tools) - 1) / 2)
ax.set_xticklabels(user_groups)

# Create title, subtitle, legend & Show graphic
ax.set_title(title, fontsize=16)
fig.suptitle(subtitle, y=1.02, fontsize=12)
handles, labels = ax.get_legend_handles_labels()
order = [0, 1, 2, 4, 3, 5]  # Reordering the legend
ax.legend(
    [handles[idx] for idx in order],
    [labels[idx] for idx in order],
    loc="upper center",
    bbox_to_anchor=(0.5, 1.15),
    ncol=6,
    title=legendtitle,
)
ax.set_ylabel(ylabel)
ax.set_ylim(ylim)  # Adjust y-axis limit to accommodate error bars

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("errorbar_34.pdf", bbox_inches="tight")

