
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Regions for education data
regions = ["North America", "Europe", "Asia", "South America", "Africa"]

# Literacy Rates (%)
literacy_rates = [99, 97, 94, 86, 70]
literacy_rates_errors = [1, 1.5, 2, 3, 4]

# Education Quality Index (out of 100)
education_quality_index = [85, 80, 78, 70, 60]
education_quality_index_errors = [3, 4, 5, 6, 7]

# Government Spending on Education (billions of USD)
gov_spending_education = [600, 500, 450, 100, 80]
gov_spending_education_errors = [50, 40, 35, 15, 10]

titles = [
    "Literacy Rates by Region",
    "Education Quality Index by Region",
    "Government Spending on Education by Region",
]
xlabels = ["% Literacy Rates", "Quality Index (out of 100)", "Billions of USD"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a subplot for each category
fig, (ax1, ax2, ax3) = plt.subplots(3, 1, figsize=(8, 10))

# Colors for bars, each plot can have its own color map or shared
colors = plt.get_cmap("cool")(np.linspace(0.2, 0.8, 5))

def add_bars_with_annotations(ax, data, errors, colors):
    bars = ax.barh(regions, data, xerr=errors, color=colors, edgecolor="black", capsize=0)
    for i, bar in enumerate(bars):
        width = bar.get_width()
        label_x_pos = bar.get_width() + errors[i] * 0.2
        ax.text(label_x_pos, bar.get_y() + bar.get_height() / 2, f"{width}", va="bottom")

# Plot Literacy Rates on the first subplot
ax1.barh(regions, literacy_rates, xerr=literacy_rates_errors, edgecolor="black", color=colors, capsize=0)
add_bars_with_annotations(ax1, literacy_rates, literacy_rates_errors, colors)
ax1.set_title(titles[0])
ax1.set_xlabel(xlabels[0])
ax1.xaxis.grid(True, linestyle="--")
ax1.spines["right"].set_visible(False)
ax1.spines["top"].set_visible(False)

# Plot Education Quality Index on the second subplot
ax2.barh(regions, education_quality_index, xerr=education_quality_index_errors, edgecolor="black", color=colors, capsize=0)
add_bars_with_annotations(ax2, education_quality_index, education_quality_index_errors, colors)
ax2.set_title(titles[1])
ax2.set_xlabel(xlabels[1])
ax2.xaxis.grid(True, linestyle="--")
ax2.spines["right"].set_visible(False)
ax2.spines["top"].set_visible(False)

# Plot Government Spending on Education on the third subplot
ax3.barh(regions, gov_spending_education, xerr=gov_spending_education_errors, edgecolor="black", color=colors, capsize=0)
add_bars_with_annotations(ax3, gov_spending_education, gov_spending_education_errors, colors)
ax3.set_title(titles[2])
ax3.set_xlabel(xlabels[2])
ax3.xaxis.grid(True, linestyle="--")
ax3.spines["right"].set_visible(False)
ax3.spines["top"].set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the figure
plt.tight_layout()
plt.savefig("errorbar_84.pdf", bbox_inches="tight")
