# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Data
cities = ["Berlin", "San Francisco", "Seoul", "Hong Kong"]
network_protocols = [
    "Latency",
    "Throughput",
    "Packet Loss",
    "Jitter",
    "Reliability",
    "Energy Efficiency",
]
# Simulated mean scores for different protocols
network_performance_means = np.array(
    [
        [30, 85, 5, 25, 95, 45],  # Berlin
        [25, 90, 10, 30, 90, 40],  # San Francisco
        [35, 80, 4, 20, 92, 50],  # Seoul
        [20, 75, 8, 22, 85, 55],  # Hong Kong
    ]
)

# Simulated standard deviations for scores (more variability)
network_performance_std = np.array(
    [
        [3, 6, 2, 4, 7, 5],  # Berlin
        [5, 7, 3, 6, 5, 6],  # San Francisco
        [4, 5, 1, 3, 6, 4],  # Seoul
        [3, 8, 2, 5, 7, 7],  # Hong Kong
    ]
)

xlabel = "Cities"
ylabel = "Performance Metrics (%)"
ylim = [0, 100]
legendtitle = "Network Protocols"
title = "Communication Network Performance Assessment"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 6))

# Vibrant color palette for communication networks theme
colors = ["#ff9999", "#66b3ff", "#99ff99", "#ffcc99", "#c2c2f0", "#ffb3e6"]

# Bar width and positions
bar_width = 0.12

# Positions of the bar groups
r = np.arange(len(cities))

# Drawing bars for different protocols
for i in range(len(network_protocols)):
    ax.bar(
        r + i * bar_width,
        network_performance_means[:, i],
        yerr=network_performance_std[:, i],
        width=bar_width,
        label=network_protocols[i],
        capsize=4,
        color=colors[i],
        edgecolor="black",
    )

# Set x-axis labels and axis properties
ax.set_xlabel(xlabel)
ax.set_xticks(r + bar_width * (len(network_protocols) / 2))
ax.set_xticklabels(cities)
ax.set_ylabel(ylabel)
ax.set_ylim(ylim)
ax.set_title(title, fontsize=16)

# Customize the legend
ax.legend(loc="lower center", bbox_to_anchor=(0.5, -0.3), title=legendtitle, ncol=3)

# Add gridlines for better readability
ax.grid(True, linestyle="--", linewidth=0.5)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("errorbar_97.pdf", bbox_inches="tight")
