# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Sample data (domain-specific: Energy)
categories = [
    "Solar",
    "Wind",
    "Hydro",
    "Nuclear",
    "Coal",
    "Natural Gas",
]
means = np.random.uniform(0.2, 0.6, len(categories))
std_devs = np.random.uniform(0.05, 0.15, len(categories))
dataset_mean = np.mean(means)

# Labels and Plot Types
label_Mean = "Mean Energy Generation Fraction"
label_Dataset_mean = "Overall Mean Generation Fraction"

# Axes Limits and Labels
ylabel_value = "Energy Production Fraction"
ylim_values = [0.1, 0.7]

title = "Energy Production Fractions by Source"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axis
fig, ax = plt.subplots(figsize=(10, 6))

# Error bar plot
ax.errorbar(
    categories,
    means,
    yerr=std_devs,
    fmt="s",
    color="purple",
    ecolor="green",
    capsize=7,
    label=label_Mean,
)

# Dataset mean line
ax.axhline(
    y=dataset_mean, color="blue", linestyle="--", linewidth=2, label=label_Dataset_mean
)

# Customizing the plot
ax.set_ylabel(ylabel_value, fontsize=12)
ax.set_xticklabels(categories, rotation=45, ha="right", fontsize=12)
ax.legend()
ax.set_ylim(ylim_values)
ax.set_title(title, fontsize=14)

# Adding grid for better readability
ax.grid(which="both", linestyle="--", linewidth=0.5)

# Annotations
ax.annotate(
    f"Mean: {dataset_mean:.2f}",
    xy=(0.5, dataset_mean),
    xytext=(0.6, dataset_mean + 0.05),
    arrowprops=dict(facecolor="black", shrink=0.05),
    fontsize=10,
    color="black",
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout to prevent clipping of tick-labels
plt.tight_layout()
plt.savefig("errorpoint_21.pdf", bbox_inches="tight")
