

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Data for plotting
import numpy as np

np.random.seed(2)
regions = [
    "North America",
    "South America",
    "Europe",
    "Asia",
    "Africa",
    "Australia",
    "Antarctica",
    "Middle East",
    "Central America",
    "Caribbean",
]
average_rainfall = [
    30.5,  # Inches
    60.4,
    40.9,
    50.6,
    45.4,
    20.2,
    8.7,
    15.8,
    70.3,
    55.5,
]
errors = [
    [5.0, -3.0],  # Error margins in Inches
    [6.2, -4.1],
    [4.5, -3.5],
    [5.3, -4.2],
    [4.0, -3.0],
    [3.5, -2.0],
    [2.5, -1.5],
    [3.0, -2.0],
    [6.0, -4.5],
    [5.0, -3.2],
]
methods = [
    "Satellite Data",
    "Ground Stations",
    "Weather Balloons",
    "Remote Sensing",
    "Rain Gauges",
    "Climate Models",
    "Aerial Surveys",
    "Hydrological Data",
    "Data Assimilation",
    "Meteorological Analysis",
]
xticks = np.arange(0.0, 80.0, 10.0)  # Scale for average rainfall
xlim = [0.0, 80.0]
xvline = 50.0  # Average annual rainfall in Inches
xvspan = [45.0, 55.0]  # Range near average
xlabel = "Average Annual Rainfall (Inches)"
ylabel = "Regions"
title = "Average Annual Rainfall Across Different Regions"
supertitle = "An Overview of Global Rainfall Distribution by Region"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting
fig, ax = plt.subplots(figsize=(12, 8))

# Color palette for the regions
colors = plt.cm.viridis(np.linspace(0, 1, len(regions)))

# Error bars with different positive and negative values
for i, (region, value, error, color) in enumerate(zip(regions, average_rainfall, errors, colors)):
    ax.errorbar(
        value,
        i,
        xerr=[[abs(error[1])], [error[0]]],
        fmt="o",
        color=color,
        ecolor='black',
        capsize=3,
    )
    ax.text(
        value,
        i - 0.15,
        r"$%.1f^{+%.2f} _{-%.2f}$" % (value, error[0], abs(error[1])),
        va="center",
        ha="center",
        fontsize=9,
        color='black'
    )

# Highlighted region with adjusted color and alpha
ax.axvspan(xvspan[0], xvspan[1], color="yellow", alpha=0.3)

# Text for methods with adjusted font size
for i, method in enumerate(methods):
    ax.text(85.0, i, method, va="center", ha="left", fontsize=11)

# Set labels and title
ax.set_yticks(range(len(regions)))
ax.set_yticklabels(regions, fontsize=11)
ax.set_xlabel(xlabel, fontsize=12)
ax.set_xlim(xlim)
ax.set_ylim(-0.5, len(regions) - 0.5)
ax.invert_yaxis()  # Invert y-axis to match the original image
ax.axvline(x=xvline, linestyle="--", color="red", linewidth=1)
# Adjust x-axis ticks and labels
ax.set_xticks(xticks)
ax.set_xticklabels([f"{x:.1f}" for x in xticks], fontsize=11)

# Set the super title and title
plt.suptitle(supertitle, fontsize=14, fontweight='bold')
plt.title(title, fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save/show the plot
plt.tight_layout(rect=[0, 0.03, 1, 0.95])
plt.savefig("errorpoint_57.pdf", bbox_inches="tight")
