

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Create a directed graph
G = nx.DiGraph()

# Add nodes with their respective colors representing community roles
nodes = {
    0: {"color": "crimson", "type": "Teacher"},
    1: {"color": "royalblue", "type": "Doctor"},
    2: {"color": "crimson", "type": "Teacher"},
    3: {"color": "darkorange", "type": "Engineer"},
    4: {"color": "royalblue", "type": "Doctor"},
    5: {"color": "darkorange", "type": "Engineer"},
    6: {"color": "crimson", "type": "Teacher"},
}
for node, attributes in nodes.items():
    G.add_node(node, color=attributes["color"], type=attributes["type"])

# Add edges with labels representing the type of social interaction
edges = [
    (0, 1, "Meetings"),
    (1, 2, "Collaborations"),
    (2, 3, "Meetings"),
    (3, 4, "Friendships"),
    (4, 5, "Collaborations"),
    (5, 6, "Meetings"),
]
for u, v, label in edges:
    G.add_edge(u, v, label=label)

# Define node positions in a circular layout
pos = nx.circular_layout(G)

# Update the title to reflect the new dataset
title = "Community Social Interaction Network"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 10))

# Draw nodes with color attribute
node_colors = [G.nodes[node]["color"] for node in G.nodes]
nx.draw_networkx_nodes(G, pos, node_color=node_colors, node_size=600)

# Draw edges with different styles
edge_styles = {
    "Meetings": {"style": "solid", "width": 2, "color": "darkred"},
    "Collaborations": {"style": "dashed", "width": 2, "color": "darkblue"},
    "Friendships": {"style": "dotted", "width": 2, "color": "darkorange"}
}
for edge in G.edges:
    u, v = edge
    label = G[u][v]["label"]
    nx.draw_networkx_edges(
        G, pos, edgelist=[(u, v)], 
        style=edge_styles[label]["style"], 
        width=edge_styles[label]["width"], 
        edge_color=edge_styles[label]["color"], 
        connectionstyle="arc3,rad=0.2", 
        arrows=True
    )

# Draw edge labels
edge_labels = {(u, v): G[u][v]["label"] for u, v in G.edges}
nx.draw_networkx_edge_labels(G, pos, edge_labels=edge_labels)

# Add node labels
nx.draw_networkx_labels(G, pos, {node: node for node in G.nodes})

# Add a title
plt.title(title)

# Create a legend for node types
legend_handles = [
    plt.Line2D(
        [0], [0],
        marker="o",
        color="w",
        markerfacecolor=color,
        label=type,
        markersize=10,
    )
    for color, type in set(
        (attributes["color"], attributes["type"]) for attributes in nodes.values()
    )
]
ax.legend(handles=legend_handles, loc="upper right")

# Remove axis
plt.axis("off")

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_44.pdf", bbox_inches="tight")

