# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
import random

np.random.seed(0)
random.seed(42)

# Create a random social media influence network graph
G = nx.random_geometric_graph(40, 0.4)

# Position the nodes based on their connections using a layout algorithm
pos = nx.spring_layout(G)  # This layout algorithm mimics the force-directed placement

# Assign random categories to nodes (e.g., influencers, followers)
categories = ["influencer", "follower"]
node_categories = {i: random.choice(categories) for i in G.nodes()}

# Define colors for categories (colors will be defined in Part 3)
node_colors = {"influencer": "dodgerblue", "follower": "lightseagreen"}

# Create a color list for nodes based on their category
node_color_list = [node_colors[node_categories[node]] for node in G.nodes()]

# Randomly select some edges to color differently
edges = list(G.edges())
highlighted_edges = np.random.choice(
    len(edges), size=int(len(edges) * 0.3), replace=False
)
highlighted_edges = [edges[i] for i in highlighted_edges]

# Labels and titles
title = "Social Media Influence Network"
legend_labels = {"influencer": "Influencers", "follower": "Followers"}

legend_title = "User Categories"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 12))

# Draw the nodes with categories
nx.draw_networkx_nodes(
    G, pos, node_size=200, node_color=node_color_list, ax=ax, edgecolors="k"
)

# Draw the edges
nx.draw_networkx_edges(G, pos, edgelist=edges, alpha=0.2, edge_color="gray", ax=ax)

# Draw the highlighted edges in a different style
nx.draw_networkx_edges(
    G,
    pos,
    edgelist=highlighted_edges,
    edge_color="orange",
    width=3,
    alpha=0.6,
    ax=ax,
    style="dashed",
)

# Title and axis settings
plt.title(title, fontsize=17)
plt.axis("off")

# Add legend manually
for cat in categories:
    ax.scatter([], [], c=node_colors[cat], label=legend_labels[cat])

ax.legend(
    scatterpoints=1,
    frameon=False,
    labelspacing=1,
    title=legend_title,
    fontsize=12,
    title_fontsize="13",
)

# Gridlines and saving the output
plt.grid(True, which="both", linestyle="--", linewidth=0.5)
plt.tight_layout()
plt.savefig("graph_72.pdf", bbox_inches="tight")
