# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
import random

np.random.seed(0)
random.seed(42)

# Number of individuals
num_individuals = 15

# Create a directed graph representing a social communication network
G = nx.DiGraph()

# Add nodes for individuals
G.add_nodes_from(range(num_individuals))

# Generate random communications between individuals
communications = [
    (random.randint(0, num_individuals - 1), random.randint(0, num_individuals - 1))
    for _ in range(50)
]
G.add_edges_from(communications)

# Position the nodes in a circular layout for visualization
pos = nx.circular_layout(G)

# Randomly select some communications to highlight
edges = list(G.edges())
highlighted_edges = np.random.choice(
    len(edges), size=int(len(edges) * 0.3), replace=False
)
highlighted_edges = [edges[i] for i in highlighted_edges]

# Text Labels
title = "Social Communication Network"

legend_labels = ["Communications", "Highlighted Communications"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 12))

# Draw the nodes
nx.draw_networkx_nodes(G, pos, node_size=800, node_color="gold", ax=ax)

# Draw the edges
nx.draw_networkx_edges(G, pos, edgelist=edges, alpha=0.5, edge_color="gray", ax=ax)

# Draw the highlighted edges
nx.draw_networkx_edges(
    G,
    pos,
    edgelist=highlighted_edges,
    edge_color="blue",
    width=2,
    alpha=0.7,
    style="dashed",
    ax=ax,
)

# Draw node labels
labels = {i: f"Person {i+1}" for i in range(num_individuals)}
nx.draw_networkx_labels(G, pos, labels, font_size=12, font_color="black", ax=ax)

# Title settings
plt.title(title, fontsize=20)
plt.axis("off")

# Add legend
blue_line = plt.Line2D([0], [0], color="blue", lw=2, linestyle="dashed")
gray_line = plt.Line2D([0], [0], color="gray", lw=2)
plt.legend([gray_line, blue_line], legend_labels, fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("graph_73.pdf", bbox_inches="tight")
