

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx
import random

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Create a random graph to simulate trade routes between 15 ancient cities
random.seed(78)
np.random.seed(78)  # For reproducibility
G = nx.erdos_renyi_graph(15, 0.2)
trade_volumes = {edge: np.random.randint(10, 500) for edge in G.edges()}  # Trade volumes between cities
nx.set_edge_attributes(G, trade_volumes, "weight")

# Use a spring layout for node positions
pos = nx.spring_layout(G, seed=78)

# Labels for nodes (ancient cities)
labels = {
    0: "Babylon", 1: "Thebes", 2: "Athens", 3: "Rome", 4: "Carthage",
    5: "Alexandria", 6: "Jerusalem", 7: "Damascus", 8: "Troy",
    9: "Sparta", 10: "Memphis", 11: "Nineveh", 12: "Ur", 
    13: "Sidon", 14: "Byblos"
}

# Draw edge labels (trade volumes)
edge_labels = nx.get_edge_attributes(G, "weight")

# Simulating population sizes for node sizes (historical significance)
population = {i: np.random.randint(20000, 500000) for i in G.nodes()}
nx.set_node_attributes(G, population, "population")

# Title and other texts
title = "Historical Trade Routes and City Populations"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(14, 12))

# Custom color scheme for nodes and edges
node_colors = "forestgreen"
edge_colors = "royalblue"

# Draw nodes with sizes based on simulated population
nx.draw(
    G, pos, node_size=[population[i] * 0.00005 for i in G.nodes()], node_color=node_colors, 
    edge_color=edge_colors, with_labels=False
)

# Draw the labels for the nodes
nx.draw_networkx_labels(
    G, pos, labels=labels, font_color="black", font_size=12, font_weight="bold"
)

# Draw the edges with varying widths based on trade volumes
nx.draw_networkx_edges(
    G, pos, width=[trade_volumes[edge] * 0.005 for edge in G.edges()], edge_color=edge_colors
)

# Draw edge labels with weights
nx.draw_networkx_edge_labels(
    G, pos, edge_labels=edge_labels, font_color="red", font_size=10
)

# Add a title and configure layout
plt.title(title, size=22)
plt.tight_layout()

# ===================
# Part 4: Saving Output
# ===================
# Save the plot
plt.savefig("graph_78.pdf", bbox_inches="tight")

