

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib
import matplotlib as mpl

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
papers = ["Paper A", "Paper B", "Paper C", "Paper D", "Paper E", "Paper F", "Paper G"]
years = ["2015", "2016", "2017", "2018", "2019", "2020", "2021"]

metrics = np.array(
    [
        [100, 150, 200, 180, 130, 210, 160],
        [140, 120, 195, 170, 140, 125, 190],
        [130, 160, 210, 220, 180, 190, 205],
        [190, 180, 175, 195, 160, 140, 200],
        [180, 150, 140, 130, 170, 160, 220],
        [160, 140, 190, 185, 170, 220, 210],
        [170, 210, 130, 180, 175, 190, 160]
    ]
)

cbarlabels = ["Citations", "Publications", "Correlation Coeff."]

# Text labels
xlabel = "Years"
ylabel = "Research Papers"
title = "Research Paper Metrics Over Years"
supertitle = "Scientific Research Performance"
legendlabels = ["High Impact", "Low Impact"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

def heatmap(
    data, row_labels, col_labels, ax=None, cbar_kw=None, cbarlabel="", **kwargs
):
    if ax is None:
        ax = plt.gca()

    if cbar_kw is None:
        cbar_kw = {}

    # Plot the heatmap
    im = ax.imshow(data, **kwargs)

    # Create colorbar
    cbar = ax.figure.colorbar(im, ax=ax, **cbar_kw)
    cbar.ax.set_ylabel(cbarlabel, rotation=-90, va="bottom")

    # Show all ticks and label them with the respective list entries.
    ax.set_xticks(np.arange(data.shape[1]), labels=col_labels)
    ax.set_yticks(np.arange(data.shape[0]), labels=row_labels)

    # Let the horizontal axes labeling appear on top.
    ax.tick_params(top=True, bottom=False, labeltop=True, labelbottom=False)

    # Rotate the tick labels and set their alignment.
    plt.setp(ax.get_xticklabels(), rotation=-30, ha="right", rotation_mode="anchor")

    # Turn spines off and create white grid.
    ax.spines[:].set_visible(False)

    ax.set_xticks(np.arange(data.shape[1] + 1) - 0.5, minor=True)
    ax.set_yticks(np.arange(data.shape[0] + 1) - 0.5, minor=True)
    ax.grid(which="minor", color="w", linestyle="-", linewidth=3)
    ax.tick_params(which="minor", bottom=False, left=False)

    return im, cbar

def annotate_heatmap(
    im,
    data=None,
    valfmt="{x:.2f}",
    textcolors=("black", "white"),
    threshold=None,
    **textkw,
):
    if not isinstance(data, (list, np.ndarray)):
        data = im.get_array()

    # Normalize the threshold to the image color range.
    if threshold is not None:
        threshold = im.norm(threshold)
    else:
        threshold = im.norm(data.max()) / 2.0

    # Set default alignment to center, but allow it to be
        # overwritten by textkw.
    kw = dict(horizontalalignment="center", verticalalignment="center")
    kw.update(textkw)

    # Get the formatter in case a string is supplied
    if isinstance(valfmt, str):
        valfmt = matplotlib.ticker.StrMethodFormatter(valfmt)

    # Loop over the data and create a `Text` for each "pixel".
    # Change the text's color depending on the data.
    texts = []
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            kw.update(color=textcolors[int(im.norm(data[i, j]) > threshold)])
            text = im.axes.text(j, i, valfmt(data[i, j], None), **kw)
            texts.append(text)

    return texts

fig, ((ax, ax2), (ax3, ax4)) = plt.subplots(2, 2, figsize=(12, 10))
plt.suptitle(supertitle, fontsize=16)

# Replicate the above example with a different font size and colormap.
im, _ = heatmap(
    metrics, papers, years, ax=ax, cmap="YlGnBu", cbarlabel=cbarlabels[0]
)
annotate_heatmap(im, valfmt="{x:d}", size=8)

# Create some new data, give further arguments to imshow
data = np.random.randint(50, 100, size=(7, 7))
experiments_y = [f"Experiment {i}" for i in range(1, 8)]
experiments_x = [f"Parameter {i}" for i in list("ABCDEFG")]
im, _ = heatmap(data, experiments_y, experiments_x, ax=ax2, vmin=0, cmap="RdYlBu", cbarlabel=cbarlabels[1])
annotate_heatmap(im, valfmt="{x:d}", size=8, threshold=75, textcolors=("black", "white"))

# Sometimes even the data itself is categorical. Here we use a
# `matplotlib.colors.BoundaryNorm` to get the data into classes
# and use this to colorize the plot, but also to obtain the class
# labels from an array of classes.
data = np.random.randn(6, 6)
categories_y = [f"Category {i}" for i in range(1, 7)]
categories_x = [f"Index {i}" for i in range(1, 7)]

qrates = list("ABCDEFG")
norm = matplotlib.colors.BoundaryNorm(np.linspace(-3.5, 3.5, 8), 7)
fmt = matplotlib.ticker.FuncFormatter(lambda x, pos: qrates[::-1][norm(x)])

im, _ = heatmap(
    data,
    categories_y,
    categories_x,
    ax=ax3,
    cmap=mpl.colormaps["coolwarm"].resampled(7),
    norm=norm,
    cbar_kw=dict(ticks=np.arange(-3, 4), format=fmt),
    cbarlabel="Research Index",
)
annotate_heatmap(
    im, valfmt=fmt, size=9, fontweight="bold", threshold=0, textcolors=("black", "white")
)

# We can nicely plot a correlation matrix. Since this is bound by -1 and 1,
# we use those as vmin and vmax. We may also remove leading zeros and hide
# the diagonal elements (which are all 1) by using a
# `matplotlib.ticker.FuncFormatter`.
corr_matrix = np.corrcoef(metrics)
im, _ = heatmap(
    corr_matrix,
    papers,
    papers,
    ax=ax4,
    cmap="coolwarm",
    vmin=-1,
    vmax=1,
    cbarlabel=cbarlabels[2],
)

def func(x, pos):
    return f"{x:.2f}".replace("0.", ".").replace("1.00", "")

annotate_heatmap(im, valfmt=matplotlib.ticker.FuncFormatter(func), size=8)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("heatmap_105.pdf", bbox_inches="tight")

