
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate internet speed data (in Mbps) across regions
data = np.array(
    [
        [12, 20, 15, 18, 10, 25],
        [22, 30, 28, 25, 24, 35],
        [18, 16, 19, 22, 15, np.nan],
        [10, 12, 11, 15, 17, 20],
    ]
)

title = "Average Internet Speed (in Mbps)"
xlabel = "Regions"
xticklabels = ["North", "South", "East", "West", "Central", "Islands"]
yticklabels = ["2017", "2018", "2019", "2020"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting the heatmap with adjusted colorbar and new theme color
# Create mask for NaN values to hatch them later
mask = np.isnan(data)

# Defining a new color palette
cmap = plt.get_cmap("Blues")
norm = plt.Normalize(vmin=np.nanmin(data), vmax=np.nanmax(data))

fig, ax = plt.subplots(figsize=(10, 8))
cax = ax.imshow(data, cmap=cmap, norm=norm)
cbar = fig.colorbar(cax, ax=ax, extend="both")

# Add hatches for NaN values
for i, j in zip(*np.where(mask)):
    ax.add_patch(
        plt.Rectangle(
            (j - 0.5, i - 0.5), 1, 1, fill=False, hatch="XX", edgecolor="black"
        )
    )

# Adding titles and labels
plt.title(title)
plt.xlabel(xlabel)

# Define the labels for x and y axis
ax.set_xticks(range(6))
ax.set_xticklabels(xticklabels, rotation=45)
ax.set_yticks(range(4))
ax.set_yticklabels(yticklabels, rotation=0)

# Add annotations
for i in range(4):
    for j in range(6):
        if not np.isnan(data[i, j]):
            if data[i, j] > np.nanmean(data) * 1.2:  # Threshold adjusted for annotation color
                ax.text(
                    j, i, f"{data[i, j]:.0f}", ha="center", va="center", color="white"
                )
            else:
                ax.text(
                    j, i, f"{data[i, j]:.0f}", ha="center", va="center", color="black"
                )

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_47.pdf", bbox_inches="tight")
