
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.colors import Normalize

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Sample data: Monthly number of cases handled for 5 courts
# Rows: Courts (Court A, Court B, Court C, Court D, Court E)
# Columns: Months (Jan, Feb, Mar, Apr, May, Jun, Jul, Aug, Sep, Oct, Nov, Dec)
data = np.array(
    [
        [120, 150, 200, 180, 220, 300, 350, 340, 290, 240, 190, 160],
        [100, 130, 180, 170, 210, 290, 320, 310, 270, 230, 180, 150],
        [90, 120, 160, 140, 200, 270, 310, 300, 260, 210, 170, 140],
        [140, 170, 220, 200, 250, 320, 360, 350, 310, 270, 220, 190],
        [150, 180, 240, 220, 270, 330, 380, 370, 330, 290, 240, 210],
    ]
)

# X and Y labels
x_labels = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"]
y_labels = ["Court A", "Court B", "Court C", "Court D", "Court E"]
xlabel = "Month"
ylabel = "Court"
title = "Monthly Number of Cases Handled by Courts"
cbar_label = "Number of Cases"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up the figure and axis
fig, ax = plt.subplots(figsize=(12, 8))

# Create a custom colormap
cmap = plt.cm.get_cmap("coolwarm")

# Set up the colormap and norm for temperatures
norm = Normalize(vmin=np.min(data), vmax=np.max(data))

# Create the scatter plot with adjusted circle sizes
for i in range(data.shape[0]):
    for j in range(data.shape[1]):
        color = cmap(norm(data[i, j]))
        circle = plt.Circle((j, i), 0.4, color=color)
        ax.add_artist(circle)
        ax.text(j, i, f"{data[i, j]:.1f}", ha="center", va="center", color="black")

# Set labels for x and y axes
ax.set_xticks(range(len(x_labels)))
ax.set_xticklabels(x_labels, rotation=45, ha="right")
ax.set_yticks(range(len(y_labels)))
ax.set_yticklabels(y_labels)

# Adding titles for the axes
ax.set_xlabel(xlabel, fontsize=14)
ax.set_ylabel(ylabel, fontsize=14)

# Set the limits of the axes
ax.set_xlim(-0.5, data.shape[1] - 0.5)
ax.set_ylim(-0.5, data.shape[0] - 0.5)

# Set the aspect of the plot to be equal and add a frame
ax.set_aspect('equal')
for spine in ax.spines.values():
    spine.set_visible(True)

# Create a colorbar
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = plt.colorbar(sm, ax=ax, orientation="vertical")
cbar.set_label(cbar_label, fontsize=12)

# Set the title of the plot
plt.title(title, fontsize=16)

# ===================
# Part 4: Saving Output
# ===================
# Save the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_60.pdf", bbox_inches="tight")
