
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import pandas as pd

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generating a similar style data for demonstration
# Assuming there are 3 philosophical time periods and 3 thinkers
time_periods = ["Ancient", "Medieval", "Modern"]
thinkers = ["Thinker 1", "Thinker 2", "Thinker 3"]
influence_data = []

# Generating random influence scores and percentage changes for the example
for thinker in thinkers:
    for period in time_periods:
        scores = np.random.randint(0, 100, size=9)
        np.random.shuffle(scores)
        percent_change = np.round(np.random.uniform(-10, 10), 2)
        for rank, score in enumerate(scores, start=1):
            influence_data.append(
                {
                    "Thinker": thinker,
                    "Period": period,
                    "Rank": rank,
                    "InfluenceScore": score,
                    "PercentChange": percent_change,
                }
            )

# Create the DataFrame
df = pd.DataFrame(influence_data)

# Pivoting the dataframe for the heatmap
df_pivot = df.pivot_table(
    index="Rank", columns=["Thinker", "Period"], values="InfluenceScore", aggfunc="first"
)

title = "Philosophical Influence Heatmap"
xlabel = "Thinker and Time Period"
ylabel = "Rank"
color_bar_label = "Influence Score"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Initialize the matplotlib figure
fig, ax = plt.subplots(figsize=(10, 10))

# Create a custom colormap for the heatmap
cmap = plt.get_cmap("coolwarm")

# Draw the heatmap
cax = ax.imshow(df_pivot, cmap=cmap)

# Decorations
plt.title(title, fontsize=16)
plt.xlabel(xlabel, fontsize=14)
plt.ylabel(ylabel, fontsize=14)

# Set x-axis and y-axis labels
ax.set_xticks(range(len(df_pivot.columns)))
ax.set_xticklabels(
    [f"{col[0]} {col[1]}" for col in df_pivot.columns], rotation=45, ha="right"
)
ax.set_yticks(range(len(df_pivot.index)))
ax.set_yticklabels(df_pivot.index)

# Add annotations
for i in range(len(df_pivot.index)):
    for j in range(len(df_pivot.columns)):
        text_color = "white" if df_pivot.iloc[i, j] < 50 else "black"
        ax.text(j, i, df_pivot.iloc[i, j], ha="center", va="center", color=text_color)

# Add colorbar to be smaller
cbar = fig.colorbar(cax, ax=ax, fraction=0.1, pad=0.04)
cbar.set_label(color_bar_label)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_76.pdf", bbox_inches="tight")
