

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Data for the two subplots representing population distribution over centuries
data1 = np.array(
    [
        [12.3, 15.2, 4.8, 7.5, 21.0],
        [8.4, 18.2, 6.1, 11.9, 30.5],
        [17.0, 10.4, 2.6, 20.5, 9.5],
        [22.5, 5.3, 8.7, 15.1, 18.4],
        [18.8, 14.5, 10.2, 9.6, 22.9],
    ]
)

data2 = np.array(
    [
        [25.3, 6.8, 18.1, 8.9, 10.9],
        [12.7, 14.5, 17.0, 22.6, 12.2],
        [20.4, 8.9, 3.3, 32.4, 14.0],
        [18.6, 7.7, 13.4, 28.3, 12.0],
        [10.5, 12.5, 20.7, 15.5, 18.8],
    ]
)


# X and Y labels
x_labels = ["Region A", "Region B", "Region C", "Region D", "Region E"]
y_labels = ["16th Century", "17th Century", "18th Century", "19th Century", "20th Century"]

# Subplot titles
titles = ["(a) Populations Over Centuries - Dataset 1", "(b) Populations Over Centuries - Dataset 2"]

# Axes Limits and Labels
xticks_values = range(len(x_labels))
yticks_values = range(len(y_labels))
colorbar_ticks = [0.1, 1, 10, 100]
yticklabels = ["0.1M", "1M", "10M", "100M"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Defining a more vibrant colormap
cmap = plt.cm.viridis

# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=0.1, vmax=100)

# Set up the figure
fig, axes = plt.subplots(
    1, 2, figsize=(20, 8), gridspec_kw={"width_ratios": [1, 1], "wspace": 0.3}
)


# Function to create a subplot
def create_subplot(ax, data, title):
    # Create the scatter plot
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            # Calculate the color based on the original value
            color = cmap(norm(data[i, j]))

            # Draw the circle with a fixed size
            circle = plt.Circle((j, i), 0.4, color=color)  # Fixed size
            ax.add_artist(circle)

            # Determine text color based on the value
            text_color = "white" if data[i, j] > 25 else "black"

            # Add the text inside the circle
            ax.text(
                j, i, f"{data[i, j]:.1f}M", ha="center", va="center", color=text_color
            )

    # Set labels for x and y axes
    ax.set_xticks(range(len(x_labels)))
    ax.set_xticklabels(x_labels, ha="center")
    ax.set_yticks(range(len(y_labels)))
    ax.set_yticklabels(y_labels, va="center")

    # Adding the title for the subplot
    ax.set_title(title, fontsize=16)

    # Set the limits of the axes; they should be one more than your data range
    ax.set_xlim(-0.5, data.shape[1] - 0.5)
    ax.set_ylim(-0.5, data.shape[0] - 0.5)

    # Set the aspect of the plot to be equal and add a frame
    ax.set_aspect("equal")
    for spine in ax.spines.values():
        spine.set_visible(True)


# Create each subplot
create_subplot(axes[0], data1, titles[0])
create_subplot(axes[1], data2, titles[1])

# Create a colorbar on the far right side of the figure
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = fig.colorbar(
    sm,
    ax=axes,
    ticks=colorbar_ticks,
    orientation="vertical",
    fraction=0.015,
    pad=0.05,
)
cbar.ax.set_yticklabels(yticklabels)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_86.pdf", bbox_inches="tight")

