
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(1)

# Data for the two subplots
data1 = np.array(
    [
        [3.2, 4.1, 5.7, 6.5, 7.8, 8.2],
        [2.8, 3.5, 4.9, 5.2, 6.3, 7.1],
        [1.9, 2.8, 3.7, 4.5, 5.1, 5.8],
        [1.3, 1.7, 2.5, 3.4, 4.0, 4.4],
        [0.7, 0.9, 1.2, 1.5, 1.8, 2.0],
    ]
)

data2 = np.array(
    [
        [7.8, 8.5, 9.1, 9.7, 10.2, 10.8],
        [6.7, 7.4, 8.0, 8.5, 9.0, 9.5],
        [5.5, 6.0, 6.5, 7.0, 7.4, 7.8],
        [4.0, 4.4, 4.8, 5.2, 5.5, 5.8],
        [2.5, 2.8, 3.0, 3.3, 3.5, 3.7],
    ]
)

# X and Y labels
x_labels = ["Early Morning", "Morning", "Noon", "Afternoon", "Evening", "Night"]
y_labels = ["Happiness", "Sadness", "Anger", "Fear", "Surprise"]

# Subplot titles
titles = ["(a) Emotions throughout the day", "(b) Average emotion levels by category"]

# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=0.1, vmax=10)

# Axes Limits and Labels
xticks_values = range(len(x_labels))
yticks_values = range(len(y_labels))
colorbar_ticks = [0.1, 1, 10]
yticklabels = ["0.1", "1", "10"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Defining the colormap from blue to red
cmap = plt.cm.coolwarm

# Set up the figure
fig, axes = plt.subplots(
    1, 2, figsize=(20, 8), gridspec_kw={"width_ratios": [1, 1], "wspace": 0.3}
)

# Function to create a subplot
def create_subplot(ax, data, title):
    # Create the scatter plot
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            # Calculate the color based on the original value
            color = cmap(norm(data[i, j]))

            # Draw the circle with a fixed size
            circle = plt.Circle((j, i), 0.4, color=color)  # Fixed size
            ax.add_artist(circle)

            # Determine text color based on the value
            text_color = "white" if data[i, j] > 5 else "black"

            # Add the text inside the circle
            ax.text(
                j, i, f"{data[i, j]:.1f}", ha="center", va="center", color=text_color
            )

    # Set labels for x and y axes
    ax.set_xticks(range(len(x_labels)))
    ax.set_xticklabels(x_labels, ha="center")
    ax.set_yticks(range(len(y_labels)))
    ax.set_yticklabels(y_labels, va="center")

    # Adding the title for the subplot
    ax.set_title(title, fontsize=16)

    # Set the limits of the axes; they should be one more than your data range
    ax.set_xlim(-0.5, data.shape[1] - 0.5)
    ax.set_ylim(-0.5, data.shape[0] - 0.5)

    # Set the aspect of the plot to be equal and add a frame
    ax.set_aspect("equal")
    for spine in ax.spines.values():
        spine.set_visible(True)


# Create each subplot
create_subplot(axes[0], data1, titles[0])
create_subplot(axes[1], data2, titles[1])

# Create a colorbar on the far right side of the figure
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = fig.colorbar(
    sm,
    ax=axes,
    ticks=colorbar_ticks,
    orientation="vertical",
    fraction=0.015,
    pad=0.05,
)
cbar.ax.set_yticklabels(yticklabels)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_87.pdf", bbox_inches="tight")
