

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Sample data: Number of scholarly articles (in hundreds) published on philosophical themes in different regions/schools
philosophy_data = np.array(
    [
        [80.3, 150.0, 110.0, 200.0, 300.0, 170.0],
        [95.0, 180.0, 145.0, 225.0, 350.0, 190.0],
        [85.5, 160.0, 120.0, 210.0, 330.0, 175.0],
        [70.0, 140.0, 105.0, 195.0, 280.0, 160.0],
        [105.0, 200.0, 130.0, 235.0, 370.0, 210.0],
    ]
)

# X and Y labels
xticklabels = ["Existentialism", "Phenomenology", "Ethics", "Epistemology", "Metaphysics", "Aesthetics"]
yticklabels = ["School A", "School B", "School C", "School D", "School E"]
x_label = "Philosophical Theme"
y_label = "Scholarly Region/School"
title = "Scholarly Articles on Philosophical Themes in Different Regions"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Custom colormap from light green to dark green
cmap = plt.cm.Greens

# Set up the figure and axis
fig, ax = plt.subplots(figsize=(10, 8))

# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=50, vmax=400)

# Create the scatter plot
for i in range(philosophy_data.shape[0]):
    for j in range(philosophy_data.shape[1]):
        # Calculate the color based on the original value
        color = cmap(norm(philosophy_data[i, j]))

        # Draw the circle
        circle = plt.Circle((j, i), 0.5, color=color)  # Adjusted size
        ax.add_artist(circle)

        # Determine text color based on the value
        text_color = "white" if philosophy_data[i, j] > 200 else "black"

        # Add the text inside the circle
        ax.text(j, i, f"{philosophy_data[i, j]:.1f}k", ha="center", va="center", color=text_color)

# Set labels for x and y axes
ax.set_xticks(range(len(xticklabels)))
ax.set_xticklabels(xticklabels, ha="center")
ax.set_yticks(range(len(yticklabels)))
ax.set_yticklabels(yticklabels, va="center")

# Adding titles for the axes and the plot
ax.set_xlabel(x_label, fontsize=14)
ax.set_ylabel(y_label, fontsize=14)
ax.set_title(title, fontsize=16, pad=20)

# Set the limits of the axes; they should be one more than your data range
ax.set_xlim(-0.5, philosophy_data.shape[1] - 0.5)
ax.set_ylim(-0.5, philosophy_data.shape[0] - 0.5)

# Set the aspect of the plot to be equal and add a frame
ax.set_aspect("equal")
for spine in ax.spines.values():
    spine.set_visible(True)

# Create a colorbar
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = plt.colorbar(sm, ax=ax, ticks=[50, 100, 200, 400], orientation="vertical")
cbar.ax.set_yticklabels(["50", "100", "200", "400"])

# Add gridlines
plt.grid(True, which="both", color="gray", linestyle="--", linewidth=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout to fit the figure size
plt.tight_layout()

# Save the plot
plt.savefig("heatmap_92.pdf", bbox_inches="tight")

