
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Sample data: a 5x6 grid representing energy consumption (in kWh)
data = np.array(
    [
        [50.3, 60.0, 70.9, 80.0, 90.4, 100.3],
        [45.2, 55.2, 65.6, 75.8, 85.3, 95.9],
        [40.8, 50.3, 60.9, 70.2, 80.5, 90.3],
        [35.0, 45.1, 55.5, 65.1, 75.1, 85.2],
        [30.0, 40.1, 50.6, 60.5, 70.6, 80.5],
    ]
)

# X and Y labels
xticklabels = ["Jan", "Feb", "Mar", "Apr", "May", "Jun"]
yticklabels = ["City A", "City B", "City C", "City D", "City E"]
x_label = "Month"
y_label = "City"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up the figure and axis
fig, ax = plt.subplots(figsize=(10, 8))

# Defining the colormap from blue (cold) to red (hot)
cmap = plt.cm.coolwarm

# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=10, vmax=120)

# Create the scatter plot
for i in range(data.shape[0]):
    for j in range(data.shape[1]):
        # Calculate the color based on the original value
        color = cmap(norm(data[i, j]))

        # Draw the circle with a fixed size
        circle = plt.Circle((j, i), 0.4, color=color)  # Fixed size
        ax.add_artist(circle)

        # Determine text color based on the value
        text_color = "white" if data[i, j] > 70 else "black"

        # Add the text inside the circle
        ax.text(j, i, f"{data[i, j]:.1f} kWh", ha="center", va="center", color=text_color)

# Set labels for x and y axes
ax.set_xticks(range(len(xticklabels)))
ax.set_xticklabels(xticklabels, ha="center")
ax.set_yticks(range(len(yticklabels)))
ax.set_yticklabels(yticklabels, va="center")

# Adding titles for the axes
ax.set_xlabel(x_label, fontsize=14)
ax.set_ylabel(y_label, fontsize=14)

# Set the limits of the axes; they should be one more than your data range
ax.set_xlim(-0.5, data.shape[1] - 0.5)
ax.set_ylim(-0.5, data.shape[0] - 0.5)

# Set the aspect of the plot to be equal and add a frame
ax.set_aspect("equal")
for spine in ax.spines.values():
    spine.set_visible(True)

# Create a colorbar
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = plt.colorbar(sm, ax=ax, ticks=[10, 30, 50, 70, 90, 110], orientation="vertical")
cbar.ax.set_yticklabels(["10", "30", "50", "70", "90", "110"])

# Add gridlines
plt.grid(True, which="both", color="gray", linestyle="--", linewidth=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout to fit the figure size
plt.tight_layout()

# Show the plot
plt.savefig("heatmap_93.pdf", bbox_inches="tight")
