# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Sample data for the environmental science domain (air quality)
pm25_levels = np.random.normal(
    loc=35, scale=15, size=1000
)  # PM2.5 levels, mean=35 µg/m³, sd=15 µg/m³
pm10_levels = np.random.normal(
    loc=60, scale=20, size=1000
)  # PM10 levels, mean=60 µg/m³, sd=20 µg/m³

# Define the number of bins and bin edges for consistent bin width
bins = np.histogram(np.hstack((pm25_levels, pm10_levels)), bins=15)[1]
labels = ["PM2.5", "PM10"]
xticks = [0, 50, 100, 150]
xlabel = "Concentration (µg/m³)"
ylabel = "Count"
title = "Distribution of Particulate Matter Concentrations in Air"
legend_title = "Pollutant Type"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
plt.figure(figsize=(6, 5))

# Create the histograms without stacking
plt.hist(
    pm25_levels,
    bins=bins,
    color="#5a9bd4",
    label=labels[0],
    edgecolor="white",
    linewidth=0.6,
    alpha=0.7,
    zorder=2,
)
plt.hist(
    pm10_levels,
    bins=bins,
    color="#f28e2b",
    label=labels[1],
    edgecolor="white",
    linewidth=0.6,
    alpha=0.7,
    zorder=3,
)

# Set the background color
plt.gca().set_facecolor("#f0f0f0")

# Set the scale of y-axis to logarithmic
plt.yscale("log")

# Set the x-axis ticks
plt.xticks(xticks)
plt.tick_params(axis="x", length=0)

# Add white grid lines and place them behind the bars (zorder=0)
plt.grid(color="white", linestyle="-", linewidth=1.5, zorder=0)

# Set the y-axis ticks and remove all line markings (spines)
plt.yticks([1, 10, 100, 1000])
plt.tick_params(axis="y", length=0)
for spine in plt.gca().spines.values():
    spine.set_visible(False)  # Remove all the spines

# remove small dash on y-axis
plt.tick_params(axis="y", which="minor", length=0)

# Set labels and title
plt.xlabel(xlabel)
plt.ylabel(ylabel)
plt.title(title)

# Add legend with title
plt.legend(title=legend_title)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout
plt.tight_layout()

# Show the plot
plt.savefig("hist_27.pdf", bbox_inches="tight")
