
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D  # Importing Line2D for creating custom legend items

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Data (Economics Domain)
time_periods = [2015, 2016, 2017, 2018, 2019, 2020, 2021]
gdp_growth = [2.5, 2.9, 2.4, 2.9, 2.3, -3.6, 5.7]
inflation_rate = [0.1, 1.3, 2.1, 2.4, 2.3, 1.2, 4.7]
unemployment_rate = [5.3, 4.9, 4.4, 3.9, 3.7, 8.1, 6.0]
interest_rate = [0.25, 0.5, 1.0, 1.5, 1.75, 0.25, 0.5]

# Axes Limits and Labels
xlabel_value = "Time Period (Years)"
ylabel_value_1 = "GDP Growth / Unemployment Rate (%)"
ylabel_value_2 = "Inflation Rate / Interest Rate (%)"
ylim_values_1 = [-5, 10]
ylim_values_2 = [-1, 6]
yticks_values_1 = range(-5, 11, 2)
yticks_values_2 = range(0, 7, 1)

# Labels
label_gdp = "GDP Growth"
label_inflation = "Inflation Rate"
label_unemployment = "Unemployment Rate"
label_interest = "Interest Rate"
title_text = "Key Economic Indicators Over Time"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plot
fig, ax1 = plt.subplots(figsize=(10, 8))

# Colors
color_gdp = "#1f77b4"
color_dow = "#ff7f0e"
color_inflation = "#2ca02c"
color_interest = "#d62728"

# GDP Growth and Unemployment Rate plots
(gdp_line,) = ax1.plot(
    time_periods,
    gdp_growth,
    "o-",
    color=color_gdp,
    label=label_gdp,
    markersize=8,
    linewidth=2.5,
)
(unemployment_line,) = ax1.plot(
    time_periods,
    unemployment_rate,
    "s--",
    color=color_dow,
    label=label_unemployment,
    markersize=8,
    linewidth=2.5,
)
ax1.set_xlabel(xlabel_value, fontsize=14)
ax1.set_ylabel(ylabel_value_1, fontsize=14, color=color_gdp)
ax1.tick_params(axis="y", labelcolor=color_gdp, direction="in", labelsize=12)
ax1.tick_params(axis="x", direction="in", labelsize=12)
ax1.set_yticks(yticks_values_1)
ax1.set_ylim(ylim_values_1)
ax1.grid(which='major', linestyle='--', linewidth='0.5', color='gray')

# Adding values to the plot for GDP Growth and Unemployment Rate
for i, txt in enumerate(gdp_growth):
    ax1.annotate(
        f"{txt}",
        (time_periods[i], txt),
        textcoords="offset points",
        xytext=(0, 10),
        ha="center",
        fontsize=10,
    )
for i, txt in enumerate(unemployment_rate):
    ax1.annotate(
        f"{txt}",
        (time_periods[i], txt),
        textcoords="offset points",
        xytext=(0, -15),
        ha="center",
        fontsize=10,
    )

# Inflation Rate and Interest Rate plots with a secondary y-axis
ax2 = ax1.twinx()
(inflation_line,) = ax2.plot(
    time_periods,
    inflation_rate,
    "^-",
    color=color_inflation,
    label=label_inflation,
    markersize=8,
    linewidth=2.5,
)
(interest_line,) = ax2.plot(
    time_periods,
    interest_rate,
    "d-.",
    color=color_interest,
    label=label_interest,
    markersize=8,
    linewidth=2.5,
)
ax2.set_ylabel(ylabel_value_2, color=color_inflation, fontsize=14)
ax2.tick_params(axis="y", labelcolor=color_inflation, direction="in", labelsize=12)
ax2.set_yticks(yticks_values_2)
ax2.set_ylim(ylim_values_2)

# Adding values to the plot for Inflation Rate and Interest Rate
for i, txt in enumerate(inflation_rate):
    ax2.annotate(
        f"{txt}",
        (time_periods[i], txt),
        textcoords="offset points",
        xytext=(0, 10),
        ha="center",
        fontsize=10,
    )
for i, txt in enumerate(interest_rate):
    ax2.annotate(
        f"{txt}",
        (time_periods[i], txt),
        textcoords="offset points",
        xytext=(0, -15),
        ha="center",
        fontsize=10,
    )

# Creating custom legend items for line styles
gdp_legend = Line2D([0], [0], color=color_gdp, linestyle="-", linewidth=2.5, label=label_gdp)
unemployment_legend = Line2D([0], [0], color=color_dow, linestyle="--", linewidth=2.5, label=label_unemployment)
inflation_legend = Line2D([0], [0], color=color_inflation, linestyle="-", linewidth=2.5, label=label_inflation)
interest_legend = Line2D([0], [0], color=color_interest, linestyle="-.", linewidth=2.5, label=label_interest)

# Adding legends
first_legend = ax1.legend(
    handles=[gdp_legend, unemployment_legend],
    loc="upper left",
    fontsize=12,
    edgecolor="black",
)
ax1.add_artist(first_legend)  # Add the first legend manually
second_legend = ax1.legend(
    handles=[inflation_line, interest_line],
    loc="upper right",
    fontsize=12,
    edgecolor="black"
)  # Add the second legend

# Adding title
plt.title(title_text, fontsize=16, fontweight='bold')

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("line_170.pdf", bbox_inches="tight")
