
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Updated steps
time_steps = np.linspace(1900, 2000, 50)  # Years from 1900 to 2000

# Generating plausible philosophical study data
popularity_analytic = 50 + np.sin(time_steps * 0.05) * 30  # Popularity of Analytic Philosophy
popularity_continental = 40 + np.cos(time_steps * 0.05) * 25  # Popularity of Continental Philosophy
num_philosophers = np.linspace(50, 200, 50) + np.random.normal(scale=10, size=50)  # Number of philosophers active
influential_works = np.maximum(0, 5 + (time_steps - 1900) * 0.05 + np.random.normal(scale=2, size=50))  # Influential works published

# Simulating standard deviations for error
std_dev = 5
popularity_analytic_std = np.full_like(popularity_analytic, std_dev)
popularity_continental_std = np.full_like(popularity_continental, std_dev)
num_philosophers_std = np.full_like(num_philosophers, std_dev)
influential_works_std = np.full_like(influential_works, std_dev)

# Axes Limits and Labels
xlabel_value = "Time (Years)"
xlim_values = [1900, 2000]
xticks_values = np.linspace(1900, 2000, 11)

ylabel_value_1 = "Popularity (%)"
ylabel_value_2 = "Influential Works Published"
yticks_values_1 = np.arange(0, 101, 20)
yticks_values_2 = np.arange(0, 31, 5)
ylim_values_1 = [0, 100]
ylim_values_2 = [-10, 20]

# Labels
popularity_analytic_label = "Popularity of Analytic Philosophy (%)"
popularity_continental_label = "Popularity of Continental Philosophy (%)"
num_philosophers_label = "Number of Philosophers Active"
influential_works_label = "Influential Works Published"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating a figure with two subplots
fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(8, 10))

# Plotting on the first subplot
ax1.plot(time_steps, popularity_analytic, "o-", color="#ff5733", label=popularity_analytic_label)
ax1.fill_between(
    time_steps,
    popularity_analytic - popularity_analytic_std,
    popularity_analytic + popularity_analytic_std,
    color="#ff5733",
    alpha=0.2,
)

ax1.plot(time_steps, popularity_continental, "s-", color="#33cc33", label=popularity_continental_label)
ax1.fill_between(
    time_steps,
    popularity_continental - popularity_continental_std,
    popularity_continental + popularity_continental_std,
    color="#33cc33",
    alpha=0.2,
)

ax2.plot(
    time_steps,
    num_philosophers,
    "^--",
    color="#3366ff",
    markerfacecolor="white",
    markeredgecolor="#3366ff",
    label=num_philosophers_label,
)
ax2.fill_between(
    time_steps,
    num_philosophers - num_philosophers_std,
    num_philosophers + num_philosophers_std,
    color="#3366ff",
    alpha=0.2,
)

ax1.set_xlabel(xlabel_value, fontsize=12)
ax1.set_ylabel(ylabel_value_1, fontsize=12)
ax1.set_xticks(xticks_values)
ax1.set_yticks(yticks_values_1)
ax1.set_xlim(xlim_values)
ax1.set_ylim(ylim_values_1)
ax1.legend(loc="upper center", frameon=False, ncol=2, bbox_to_anchor=(0.5, 1.15))
ax1.grid(
    True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5
)
ax1.set_facecolor("#f9f9f9")

# Plotting on the second subplot
ax2.plot(
    time_steps, influential_works, "*-", color="#e67300", label=influential_works_label
)
ax2.fill_between(
    time_steps,
    influential_works - influential_works_std,
    influential_works + influential_works_std,
    color="#e67300",
    alpha=0.2,
)
ax2.set_xlabel(xlabel_value, fontsize=12)
ax2.set_ylabel(ylabel_value_2, fontsize=12)
ax2.set_xticks(xticks_values)
ax2.set_yticks(yticks_values_2)
ax2.set_xlim(xlim_values)
ax2.set_ylim(ylim_values_2)
ax2.legend(loc="upper center", frameon=False, ncol=2, bbox_to_anchor=(0.5, 1.15))
ax2.grid(
    True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5
)
ax2.set_facecolor("#f9f9f9")

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("line_193.pdf", bbox_inches="tight")
