

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(1)

# Generate distinct data sets for each subplot
years = np.array([1980, 1990, 2000, 2010, 2020, 2030])
datasets = {
    "GDP Growth": np.abs(np.random.normal(loc=2, scale=1, size=len(years))),
    "Inflation Rate": np.abs(np.random.normal(loc=3, scale=1.5, size=len(years))),
    "Unemployment Rate": np.abs(np.random.normal(loc=5, scale=2, size=len(years))),
    "Stock Market Index": np.abs(np.random.normal(loc=3000, scale=1000, size=len(years))),
}

# Assign each dataset to a subplot
plot_order = ["GDP Growth", "Inflation Rate", "Unemployment Rate", "Stock Market Index"]

# Axes Limits and Labels
xlabel_value = "Year"
ylabel_value = "Percentage / Index Value"
ylim_values = [0.1, 10000]

# Text
text = "Economic Indicator Peak"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size and define colors, markers, and linestyles
fig, axs = plt.subplots(2, 2, figsize=(12, 8))  # Use a 2x2 subplot grid
colors = ["#1b9e77", "#d95f02", "#7570b3", "#e7298a"]
markers = ["o", "s", "^", "d"]
linestyles = ["-", "--", ":", "-."]

# Plot data in each subplot
for i, (ax, key) in enumerate(zip(axs.flat, plot_order)):
    for j, data_key in enumerate(plot_order):
        if key == data_key:
            ax.loglog(
                years,
                datasets[data_key],
                linestyle=linestyles[j],
                marker=markers[j],
                color=colors[j],
                label=data_key,
                markersize=8,
            )
        else:
            ax.loglog(
                years,
                datasets[data_key],
                linestyle=linestyles[j],
                marker=markers[j],
                color=colors[j],
                label=data_key,
                markersize=8,
                alpha=0.8,
            )  # Faded other lines

    ax.set_xlabel(xlabel_value)
    ax.set_ylabel(ylabel_value)
    ax.set_ylim(ylim_values)  # Ensure y-axis ranges don't clip data
    ax.set_title(f"Economic Indicator {i+1}")
    ax.legend(loc="right",bbox_to_anchor=(1, 0.55), fontsize=10)

    # Annotations to explain features, only on primary dataset for clarity
    if i == 0 or i == 3:
        ax.annotate(
            text,
            xy=(years[-2], datasets[key][-2]),
            xytext=(years[-4], datasets[key][-2] / 10),
            arrowprops=dict(arrowstyle="->", color="navy"),
            textcoords="data",
        )
    else:
        ax.annotate(
            text,
            xy=(years[-2], datasets[key][-2]),
            xytext=(years[-4], datasets[key][-2] * 1.5),
            arrowprops=dict(arrowstyle="->", color="navy"),
            textcoords="data",
        )

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout to prevent overlap and ensure clarity
plt.tight_layout()
plt.savefig("line_203.pdf", bbox_inches="tight")

