

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate distinct data sets for each subplot related to philosophical trends
years = np.array([2000, 2005, 2010, 2015, 2020, 2025])
datasets = {
    "Historical Influence": np.logspace(0.1, 2, len(years)),  # Logarithmic growth 
    "Philosophical Publications": np.linspace(5, 50, len(years)),  # Linear increase
    "Philosophical Conferences": np.array([2, 5, 10, 20, 35, 50]),  # Increasing trend
    "Public Interest in Philosophy": np.array([10, 30, 55, 75, 85, 90]) / 100,  # S-curve
}

# Assign each dataset to a subplot
plot_order = ["Historical Influence", "Philosophical Publications", "Philosophical Conferences", "Public Interest in Philosophy"]

# Axes Limits and Labels
xlabel_value = "Year"

ylabel_values = {
    "Historical Influence": "Index Value",
    "Philosophical Publications": "Number of Publications",
    "Philosophical Conferences": "Number of Conferences",
    "Public Interest in Philosophy": "Percentage (%)",
}
ylim_values = [-15, 100]

# Text
annotations = {
    "Historical Influence": "Notable Peaks",
    "Philosophical Publications": "Steady Rise",
    "Philosophical Conferences": "Increasing Events",
    "Public Interest in Philosophy": "Growing Interest",
}

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size and define colors, markers, and linestyles
fig, axs = plt.subplots(2, 2, figsize=(14, 10))  # Use a 2x2 subplot grid
colors = ["royalblue", "goldenrod", "seagreen", "darkorchid"]
markers = ["o", "s", "^", "d"]
linestyles = ["-", "--", ":", "-."]

# Plot data in each subplot
for i, (ax, key) in enumerate(zip(axs.flat, plot_order)):
    for j, data_key in enumerate(plot_order):
        if key == data_key:
            ax.plot(
                years,
                datasets[data_key],
                linestyle=linestyles[j],
                marker=markers[j],
                color=colors[j],
                label=data_key,
                markersize=8,
            )
        else:
            ax.plot(
                years,
                datasets[data_key],
                linestyle=linestyles[j],
                marker=markers[j],
                color=colors[j],
                label=data_key,
                markersize=8,
                alpha=0.8,
            )  # Faded other lines

    ax.set_xlabel(xlabel_value)
    ax.set_ylabel(ylabel_values[key])
    ax.set_ylim(ylim_values)  # Ensure y-axis ranges don't clip data
    ax.set_title(f"{key} Over Years")
    ax.legend()

    # Annotations to explain features, only on primary dataset for clarity
    annotation_text = annotations[key]
    ax.annotate(
        annotation_text,
        xy=(years[-1], datasets[key][-1]),
        xytext=(years[3], datasets[key][-1]-10),
        arrowprops=dict(arrowstyle="->", color="navy"),
        textcoords="data",
    )

# Adjust layout to prevent overlap and ensure clarity
plt.suptitle("Philosophical Trends Over Years", fontsize=16)
plt.tight_layout(rect=[0, 0, 1, 0.96])
plt.savefig("line_205.pdf", bbox_inches="tight")

