

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate new data representing religious trends over time
time_points = np.linspace(1970, 2020, 50)  # Representing years

# Simulate percentage attending religious services monthly (declining trend)
religious_service_attendance = -0.3 * (time_points - 1970) + 60 + np.random.normal(0, 2, time_points.shape)

# Simulate percentage identifying as religiously unaffiliated (increasing trend)
religiously_unaffiliated = 0.4 * (time_points - 1970) + 10 + np.random.normal(0, 2, time_points.shape)

# Calculate uncertainty bounds for visual emphasis
upper_bound_service = religious_service_attendance + 5
lower_bound_service = religious_service_attendance - 5
upper_bound_unaffiliated = religiously_unaffiliated + 5
lower_bound_unaffiliated = religiously_unaffiliated - 5

# Axes Limits and Labels
xlabel_value = "Year"
xlim_values = [1970, 2020]
xticks_values = np.arange(1970, 2021, 5)

ylabel_value = "Percentage (%)"
ylim_values = [0, 70]
yticks_values = np.linspace(0, 70, 8)

# Labels
label_service = "Monthly Religious Service Attendance (%)"
label_unaffiliated = "Religiously Unaffiliated (%)"

# Title
title = "Trends in Religious Service Attendance and Unaffiliation"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size and create a single plot
fig, ax = plt.subplots(figsize=(10, 6))

# Plot the religious service attendance with filled uncertainty
ax.fill_between(
    time_points,
    lower_bound_service,
    upper_bound_service,
    color="lightgreen",
    alpha=0.3,
)
ax.plot(
    time_points,
    religious_service_attendance,
    label=label_service,
    color="green",
    linestyle="-",
    linewidth=2,
    marker='o',
)

# Plot the religiously unaffiliated percentage with filled uncertainty
ax.fill_between(
    time_points,
    lower_bound_unaffiliated,
    upper_bound_unaffiliated,
    color="lightblue",
    alpha=0.3,
)
ax.plot(
    time_points,
    religiously_unaffiliated,
    label=label_unaffiliated,
    color="blue",
    linestyle="--",
    linewidth=2,
    marker='x',
)

# Customize the axes and grid
ax.set_title(title, fontsize=16)
ax.set_xlabel(xlabel_value, fontsize=14)
ax.set_ylabel(ylabel_value, fontsize=14)
ax.set_xlim(xlim_values)
ax.set_ylim(ylim_values)
ax.set_xticks(xticks_values)
ax.set_yticks(yticks_values)
ax.grid(True, linestyle='--', alpha=0.7)

# Add a legend to the plot
ax.legend(loc="upper right", frameon=True, fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
# Enhance layout and display
plt.tight_layout()
plt.savefig("line_211.pdf", bbox_inches="tight")
