
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# New dataset: Technology metrics
weeks = list(range(1, 29))
bug_count_data = np.random.poisson(lam=20, size=len(weeks)) - np.linspace(0, 10, len(weeks))
user_retention_data = np.linspace(0.4, 0.8, len(weeks)) + np.random.normal(scale=0.05, size=len(weeks))
deployment_frequency_data = np.linspace(1, 5, len(weeks)) + np.random.normal(scale=0.5, size=len(weeks))

# Extracted variables
xlabel = "Weeks"
ylabel_bug_count = "Bug Count"
ylabel_user_retention = "User Retention Rate"
xlim = (0, 30)
ylim_bug_count = (0, 30)
ylim_user_retention = (0, 1)
title = "Weekly Technology Metrics Analysis"
line_label_bug_count = "Bug Count"
line_label_user_retention = "User Retention Rate"
line_label_deployment_frequency = "Deployment Frequency"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis
fig, ax1 = plt.subplots(figsize=(10, 6))

# Plot the bug count data on the primary y-axis
color = "#1f77b4"  # Blue
ax1.set_xlabel(xlabel)
ax1.set_ylabel(ylabel_bug_count, color=color)
(line1,) = ax1.plot(
    weeks,
    bug_count_data,
    color=color,
    marker="o",
    linestyle="-",
    clip_on=False,
    zorder=10,
    linewidth=2,
    label=line_label_bug_count,
)
ax1.tick_params(axis="y", labelcolor=color)
ax1.set_ylim(ylim_bug_count)

# Create a secondary y-axis for the user retention data
ax2 = ax1.twinx()
color = "#ff7f0e"  # Orange
ax2.set_ylabel(ylabel_user_retention, color=color)
(line2,) = ax2.plot(
    weeks,
    user_retention_data,
    color=color,
    marker="s",
    linestyle="--",
    clip_on=False,
    zorder=10,
    linewidth=2,
    label=line_label_user_retention,
)
ax2.tick_params(axis="y", labelcolor=color)
ax2.set_ylim(ylim_user_retention)

# Add a third line for deployment frequency data
color = "#2ca02c"  # Green
(line3,) = ax1.plot(
    weeks,
    deployment_frequency_data,
    color=color,
    marker="^",
    linestyle=":",
    linewidth=2,
    clip_on=False,
    zorder=10,
    label=line_label_deployment_frequency,
)

# Customize the plot with a title, grid, and background color
fig.patch.set_facecolor("#f4f4f4")
ax1.set_facecolor("#e5f5f9")
ax2.set_facecolor("#f9e5e6")
ax1.set_xlim(xlim)
ax1.tick_params(axis="both", which="both", length=0)
ax2.tick_params(axis="both", which="both", length=0)
lines = [line1, line2, line3]
labels = [line.get_label() for line in lines]
fig.legend(
    lines, labels, loc="upper center", bbox_to_anchor=(0.5, 0.95), ncol=3, frameon=False
)
# Set the title and display the plot
plt.title(title, y=1.05)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better spacing and display
plt.tight_layout()
plt.savefig("line_256.pdf", bbox_inches="tight")
