

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generate new data for the plot
weeks = np.arange(1, 29)
aqi = np.clip(
    50 + 15 * np.sin(weeks * 0.2) + np.random.normal(0, 5, len(weeks)), 30, 100
)
co2_levels = np.clip(
    400 + 60 * np.cos(weeks * 0.2) + np.random.normal(0, 20, len(weeks)), 350, 450
)
pm25_levels = np.clip(
    35 + 10 * np.sin(weeks * 0.3) + np.random.normal(0, 5, len(weeks)), 20, 50
)

# Axes Limits and Labels
xlabel_value = "Weeks"
ylabel_value = "Measurement Units"

# Labels
label_aqi = "Air Quality Index (AQI)"
label_pm25 = "PM2.5 Levels (µg/m³)"
label_legend = ["Air Quality Index (AQI)", "CO2 Levels (ppm)", "PM2.5 Levels (µg/m³)"]

# Titles
title = "Air Quality Trends Across Weeks"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis
fig, ax = plt.subplots(figsize=(12, 8))

# Plot the AQI data
ax.plot(weeks, aqi, label=label_aqi, color="steelblue", linewidth=3)

# Plot the CO2 levels data with a gradient color
for i in range(len(weeks) - 1):
    ax.plot(
        weeks[i : i + 2],
        co2_levels[i : i + 2],
        linestyle="-",
        linewidth=3,
        color=plt.cm.Greens(i / len(weeks)),
    )

# Plot the PM2.5 levels data
ax.plot(weeks, pm25_levels, label=label_pm25, color="darkorange", linestyle="--", linewidth=3)

# Customize the plot with labels, title, and legend
ax.set_title(title, fontsize=18)
ax.set_xlabel(xlabel_value, fontsize=14)
ax.set_ylabel(ylabel_value, fontsize=14)

# Add a legend to the plot
custom_lines = [
    plt.Line2D([0], [0], color="steelblue", lw=4),
    plt.Line2D([0], [0], color="green", lw=4),
    plt.Line2D([0], [0], color="darkorange", lw=4, linestyle="--"),
]
ax.legend(
    custom_lines,
    label_legend,
    fontsize=12,
)

# Add a grid to the plot
ax.set_facecolor("whitesmoke")
ax.grid(True, which="both", linestyle=":", linewidth=0.75, color="gray")

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better spacing and display
plt.tight_layout()

# Show the plot
plt.savefig("line_258.pdf", bbox_inches="tight")

