
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Data for the plot
import numpy as np

np.random.seed(0)
years = np.arange(2010, 2020)
co2_emissions = np.clip(
    np.linspace(5.5, 6.5, len(years)) + np.random.normal(0, 0.1, len(years)),
    5,
    7,
) * 1e9  # in Metric Tons
avg_temperature = np.clip(
    np.linspace(14, 15, len(years)) + np.random.normal(0, 0.2, len(years)),
    13,
    16,
)  # in Degrees Celsius

# Axes Limits and Labels
xlabel_value = "Year"
ylabel_value_1 = "CO2 Emissions (Metric Tons)"
ylabel_value_2 = "Average Temperature (°C)"

# Labels
label_1 = "CO2 Emissions"
label_2 = "Average Temperature"

label = "Temperature Threshold"

# Threshold value
threshold = 15.5

# Annotation highlight
highlight = [2011, 2014, 2018]  # Years to highlight

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis
fig, ax = plt.subplots(figsize=(10, 6))

# Plotting CO2 emissions
color_co2 = "#ff7f0e"
color_temp = "#1f77b4"
highlight_color = "red"

(co2_line,) = ax.plot(
    years,
    co2_emissions,
    "o-",
    color=color_co2,
    label=label_1,
    markersize=8,
    linewidth=2,
)
ax.set_xlabel(xlabel_value, fontsize=14)
ax.set_ylabel(ylabel_value_1, fontsize=14)
ax.tick_params(axis="y", direction="in", labelsize=12)
ax.tick_params(axis="x", direction="in", labelsize=12)

# Create a secondary y-axis for Average Temperature
(temp_line,) = ax.plot(
    years,
    avg_temperature,
    "s--",
    color=color_temp,
    label=label_2,
    markersize=8,
    linewidth=2,
)
ax.set_ylim(bottom=4.5e9, top=7.5e9)
ax.set_xlim(left=2009, right=2020)

# Add a legend to the plot
ax.axhline(y=5.5e9, color="grey", linestyle="--", linewidth=1.5)
ax_right = ax.twinx()
ax_right.plot(years, avg_temperature, 's--', color=color_temp)
ax_right.set_ylabel(ylabel_value_2, fontsize=14)
ax_right.axhline(y=threshold, color=highlight_color, linestyle="-", linewidth=1.5, label=label)
for year in highlight:
    ax_right.plot(
        year, avg_temperature[np.where(years == year)], "ro"
    )
    ax_right.annotate(
        f"Highlight {year}",
        (year, avg_temperature[np.where(years == year)]),
        textcoords="offset points",
        xytext=(0, 10),
        ha="center",
    )

# Customize the plot with labels, title, and legend
ax.legend(fontsize=12)

# Add a grid to the plot
ax.grid(True, linestyle="--", alpha=0.6)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better spacing and display
plt.tight_layout()
plt.savefig("line_276.pdf", bbox_inches="tight")
