
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from mpl_toolkits.mplot3d import Axes3D

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate synthetic data for temperature changes over time
def temperature_change(t):
    return 20 + 10 * np.sin(2 * np.pi * t / 12) + np.random.normal(0, 2, len(t))

years = np.arange(0.0, 60.0, 1.0)
months = np.arange(0.0, 60.0, 0.25)
weeks = np.arange(0.0, 12.0, 0.1)

# Generate synthetic geographic data for temperature distribution
X = np.arange(-5, 5, 0.25)
Y = np.arange(-5, 5, 0.25)
X, Y = np.meshgrid(X, Y)
Z = 20 + 15 * np.sin(np.sqrt(X**2 + Y**2))

# Axes Limits and Labels
ylabel_value_1 = "Temperature (°C)"
xlabel_value_1 = "Time (Months)"

zlim_values = [-10, 40]
xlabel_value_2 = "Longitude"
ylabel_value_2 = "Latitude"
zlabel_value_2 = "Temperature (°C)"

# Additional label and legend settings
title_1 = "Temperature Change Over Time"
title_2 = "Geographical Temperature Distribution"
legend_labels = ["Yearly", "Monthly"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up a figure twice as tall as it is wide
fig = plt.figure(figsize=(8, 10))

# First subplot
ax1 = fig.add_subplot(2, 1, 1)
ax1.plot(years, temperature_change(years), 'o', color='darkblue', markerfacecolor='red', label=legend_labels[0])
ax1.plot(months, temperature_change(months), '--', color='orange', markerfacecolor='green', label=legend_labels[1])
ax1.grid(True)
ax1.set_ylabel(ylabel_value_1)
ax1.set_xlabel(xlabel_value_1)
ax1.set_title(title_1)
ax1.legend()

# Second subplot
ax2 = fig.add_subplot(2, 1, 2, projection="3d")
surf = ax2.plot_surface(X, Y, Z, rstride=1, cstride=1, linewidth=0, antialiased=False, cmap='viridis')
ax2.set_zlim(zlim_values)
ax2.set_xlabel(xlabel_value_2)
ax2.set_ylabel(ylabel_value_2)
ax2.set_zlabel(zlabel_value_2)
ax2.set_title(title_2)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
# Show the plot
plt.savefig("multidiff_36.pdf", bbox_inches="tight")
