
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.gridspec as gridspec
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Sample data to represent different communication platforms
platforms = ["Email", "SMS", "Social Media", "Phone Call", "Video Call"]
response_times = [5.0, 2.0, 15.0, 3.0, 20.0]  # Average response time in minutes for each platform
usage_frequency = [8.0, 7.5, 9.0, 6.5, 5.5]  # Usage frequency index out of 10

# Data for violin plot; customer satisfaction scores (1-10 scale)
satisfaction_data = np.random.normal(loc=[7, 8, 6, 7, 6], scale=0.5, size=(50, 5))

ax0xlabel = "Average Response Time (minutes)"
ax0ylabel = "Usage Frequency Index"
ax0title = "Communication Platform Usage vs Response Time"
ax1xticks = range(len(platforms))
ax1xlabel = "Communication Platforms"
ax1ylabel = "Customer Satisfaction"
ax1title = "Customer Satisfaction Distribution Across Platforms"
x = np.linspace(1, 10, 300)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create 1x2 subplot layout
fig = plt.figure(figsize=(10, 6))
gs = gridspec.GridSpec(1, 2, width_ratios=[1, 2])

# Scatter plot on the left
ax0 = fig.add_subplot(gs[0])
sc = ax0.scatter(
    response_times, usage_frequency, s=100, c=np.linspace(0.1, 0.9, len(platforms)), cmap="autumn"
)
for i, platform in enumerate(platforms):
    ax0.text(response_times[i], usage_frequency[i] + 0.05, platform, fontsize=9)
ax0.set_xlabel(ax0xlabel)
ax0.set_ylabel(ax0ylabel)
ax0.set_title(ax0title)

# Violin plot on the right
ax1 = fig.add_subplot(gs[1])

# Creating half-violins
for i, platform in enumerate(platforms):
    kde = gaussian_kde(satisfaction_data[:, i])
    y = kde(x)
    max_y = max(y)
    ax1.fill_betweenx(
        x, -y / max_y * 0.5 + i, i, color="lightgreen", alpha=0.5)
    ax1.fill_betweenx(
        x, y / max_y * 0.5 + i, i, color="green", alpha=0.5)

ax1.set_xticks(ax1xticks)
ax1.set_xticklabels(platforms)
ax1.set_xlabel(ax1xlabel)
ax1.set_ylabel(ax1ylabel)
ax1.set_title(ax1title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
# Show plot
plt.savefig("multidiff_42.pdf", bbox_inches="tight")

