

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Data for the first plot (bar plot with white circles)
tech_types = [
    "AI",
    "Blockchain",
    "Cloud",
    "Cybersecurity",
    "IoT",
    "AR/VR",
    "5G",
    "Edge Computing",
    "Quantum Computing",
    "Robotics",
    "Nanotech",
    "Biotech",
    "AI",
    "Blockchain",
    "Cloud",
    "Cybersecurity",
]
n_categories = len(tech_types)

tech_efficiency_actual = np.random.uniform(70, 120, n_categories)
tech_efficiency_predicted = tech_efficiency_actual - np.random.uniform(5, 15, n_categories)
sorted_indices = np.argsort((tech_efficiency_actual + tech_efficiency_predicted) / 2)
categories = np.array(tech_types)[sorted_indices]
tech_efficiency_actual = tech_efficiency_actual[sorted_indices]
tech_efficiency_predicted = tech_efficiency_predicted[sorted_indices]

regions = ["Region X", "Region Y", "Region Z", "Region W"]

# Data for the second plot (error bar plot with legend)
for i in range(len(regions)):
    performance_metrics = np.random.randint(-300, 300, size=5)
    actual_performance = np.random.randint(100, 400, size=5)
    error = np.random.randint(15, 25, size=5)

# Data for the third plot (scatter plot with trend line and correct text)
aggregate_performances = [
    np.linspace(150, 220, 5) - 100 * i + np.random.randint(-30, 30, 5)
    for i in range(len(regions))
]
performance_accuracies = [
    np.linspace(65, 95, 5) - 15 * i + np.random.randint(-20, 20, 5)
    for i in range(len(regions))
]
scatterlabels = ["Actual Performance", "Predicted Performance"]
xlabels = ["Performance Rating", "Aggregate Performance Rating", "Aggregate Performance Rating"]
ylabels = ["Technology Type", "Actual Performance - Predicted Performance", "Accuracy (%)"]

ax2textlabel = "Region: Region W"
title = "Technology Performance Analysis"
text_j = 250
text_i = 50
legendtitle="Measurement"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Adjusting figure size to match the original image's dimensions
fig, axs = plt.subplots(1, 3, figsize=(10, 4))

# Colors and assignments for the first plot
colors = ["#1f77b4", "#ff7f0e", "#2ca02c", "#d62728"]

# First plot (bar plot with white circles)
axs[0].grid(axis="x", zorder=0)
# Plotting the lines first
for i in range(n_categories):
    color = colors[i // 4]
    axs[0].plot(
        [tech_efficiency_predicted[i], tech_efficiency_actual[i]], [i, i], color=color, lw=6, zorder=1
    )  # Colored lines for each category
# Plotting the points
axs[0].scatter(
    tech_efficiency_actual,
    range(n_categories),
    color="black",
    label=scatterlabels[0],
    zorder=2,
    s=80,
)  # Black dots for 'Actual Performance'
axs[0].scatter(
    tech_efficiency_predicted,
    range(n_categories),
    color="white",
    edgecolors="black",
    label=scatterlabels[1],
    zorder=2,
    s=80,
)  # White dots for 'Predicted Performance'
axs[0].legend(loc="lower right", title=legendtitle)
axs[0].set_xlabel(xlabels[0])
axs[0].set_ylabel(ylabels[0])
axs[0].set_yticks(range(n_categories))
axs[0].set_yticklabels(categories[::-1])

# Second plot (error bar plot with legend)
for i in range(len(regions)):
    axs[1].errorbar(
        performance_metrics,
        actual_performance,
        yerr=error,
        xerr=error + 10,
        fmt="o",
        capsize=3,
        color=colors[i],
    )
axs[1].set_xlabel(xlabels[1])
axs[1].set_ylabel(ylabels[1])
axs[1].grid()

# Third plot (scatter plot with trend line and correct text)
for i in range(len(regions)):
    aggregate_performance = aggregate_performances[i]
    performance_accuracy = performance_accuracies[i]
    error = np.random.randint(5, 10, size=5)
    axs[2].errorbar(
        aggregate_performance,
        performance_accuracy,
        yerr=error,
        xerr=error + 10,
        fmt="o",
        capsize=3,
        color=colors[i],
    )
# Plotting the trend line
axs[2].plot(
    np.linspace(-350, 250, 5), np.linspace(50, 100, 5), color="black", lw=2, ls="-"
)
axs[2].set_xlabel(xlabels[2])
axs[2].set_ylabel(ylabels[2])
# set box edge color
axs[2].text(
    text_j,
    text_i,
    ax2textlabel,
    fontsize=8,
    color="black",
    ha="right",
    va="bottom",
    bbox=dict(facecolor="white", edgecolor="black"),
)
axs[2].grid()

# add legend to the whole figure and set colormaps
legend_handles = [
    mlines.Line2D(
        [], [], color=color, marker="s", linestyle="None", markersize=10, label=label
    )
    for label, color in zip(regions, colors)
]
fig.legend(
    handles=legend_handles,
    loc="upper center",
    title=title,
    ncol=len(regions),
    bbox_to_anchor=(0.5, 1.15),
    facecolor="white",
)

# ===================
# Part 4: Saving Output
# ===================
# Show plot
plt.tight_layout()
plt.savefig("multidiff_51.pdf", bbox_inches="tight")

