
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Data for the bar chart
import numpy as np
energy_sources = [
    "Solar Power (R1)",
    "Wind Energy (R2)",
    "Hydroelectricity (R3)",
    "Biomass (R4)",
    "Geothermal (R5)",
    "Hydrogen (R6)",
    "Tidal Energy (R7)",
]
efficiency_scores = [20.5, 45.0, 60.5, 35.0, 50.0, 65.5, 55.0]

# Data for the heatmap, representing effectiveness correlation matrix
renewable_corr = np.array(
    [
        [1.00, 0.45, 0.65, 0.50, 0.40, 0.55, 0.60],
        [0.45, 1.00, 0.70, 0.55, 0.60, 0.75, 0.80],
        [0.65, 0.70, 1.00, 0.65, 0.65, 0.78, 0.85],
        [0.50, 0.55, 0.65, 1.00, 0.68, 0.65, 0.70],
        [0.40, 0.60, 0.65, 0.68, 1.00, 0.55, 0.65],
        [0.55, 0.75, 0.78, 0.65, 0.55, 1.00, 0.79],
        [0.60, 0.80, 0.85, 0.70, 0.65, 0.79, 1.00],
    ]
)

# Titles and labels
ax1title = "Renewable Energy Efficiency Scores (%)"
ax2title = "Effectiveness Correlation"
ax1xlabel = "Efficiency Score"
ax2xlabel = "Energy Source"
ax1xlim = [0, 70]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axes
fig, (ax1, ax2) = plt.subplots(
    1, 2, figsize=(12, 7), gridspec_kw={"width_ratios": [1, 1.25]}
)

# Bar chart
y_pos = np.arange(len(energy_sources))
bar_color = "#3b7dd8"
text_color = "white"
ax1.barh(y_pos, efficiency_scores, color=bar_color)
ax1.set_yticks(y_pos)
ax1.set_yticklabels(energy_sources, fontsize=10)
ax1.invert_yaxis()  # labels read top-to-bottom
ax1.set_title(ax1title, fontsize=12)
ax1.set_xlabel(ax1xlabel, fontsize=10)
ax1.set_xlim(ax1xlim)
for i, v in enumerate(efficiency_scores):
    ax1.text(v - 5.0, i, "{:.1f}".format(v), color=text_color, va="center")

# Heatmap
im = ax2.imshow(renewable_corr, cmap="Blues", aspect="auto")

# Show all ticks and label them with the respective list entries
ax2.set_xticks(np.arange(len(energy_sources)))
ax2.set_xticklabels(energy_sources, fontsize=10)
ax2.yaxis.set_visible(False)

# Rotate the tick labels and set their alignment
plt.setp(ax2.get_xticklabels(), rotation=45, ha="right", rotation_mode="anchor")

# Loop over data dimensions and create text annotations
for i in range(len(energy_sources)):
    for j in range(len(energy_sources)):
        text = ax2.text(
            j, i, "{:.2f}".format(renewable_corr[i, j]), ha="center", va="center", color="black"
        )

ax2.set_title(ax2title, fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the figure
plt.tight_layout()
plt.savefig("multidiff_56.pdf", bbox_inches="tight")
