
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Data for bar chart (Average Daily Patient Visits)
cities = ["Hospital A", "Hospital B", "Hospital C", "Hospital D"]
avg_daily_patients = [150, 230, 175, 130]  # average daily patient visits

# Data for boxplot (Age distribution of patients)
age_data = [np.random.normal(loc, 10, 100) for loc in [35, 40, 30, 25]]

# Thresholds and labels
patient_threshold_upper = 220
patient_threshold_lower = 140
ylabels = ["Average Daily Patients", "Patient Age Distribution"]
ax2_hlines_labels = ["Upper patient threshold", "Lower patient threshold"]
ax2_ylim = [-10, 60]

# Additional parameters
xlabel = "Hospitals"
title = "Healthcare Analysis by Hospital"
suptitle = "Hospital-wise Average Daily Patients and Age Distribution"
legend_labels = ["Upper Patient Threshold", "Lower Patient Threshold"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and subplots with shared x-axis
fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(8, 8), sharex=True)

# Bar chart
ax1.bar(
    cities,
    avg_daily_patients,
    hatch="o",
    color="#ff9999",
    edgecolor="black",
    width=0.6,
)
ax1.set_ylabel(ylabels[0])

# Boxplot chart
ax2.boxplot(age_data, positions=range(len(cities)), widths=0.6, patch_artist=True,
             boxprops=dict(facecolor='#66b3ff', color='black'),
             capprops=dict(color='black'),
             whiskerprops=dict(color='black'),
             flierprops=dict(color='black', markeredgecolor='black'),
             medianprops=dict(color='black'))
ax2.hlines(
    patient_threshold_upper,
    xmin=-0.5,
    xmax=len(cities) - 0.5,
    colors="orange",
    linestyles="dashed",
    label=ax2_hlines_labels[0],
)
ax2.hlines(
    patient_threshold_lower,
    xmin=-0.5,
    xmax=len(cities) - 0.5,
    colors="green",
    linestyles="dashed",
    label=ax2_hlines_labels[1],
)
ax2.fill_between(
    [-0.5, len(cities) - 0.5],
    patient_threshold_lower,
    patient_threshold_upper,
    color="grey",
    alpha=0.15,
)
ax2.set_ylabel(ylabels[1])
ax2.set_xlabel(xlabel)
ax2.yaxis.set_major_formatter(plt.FuncFormatter(lambda x, _: f"{x:.0f}"))
ax2.set_ylim(ax2_ylim)
ax2.set_xticklabels(cities)
ax2.legend(loc="upper right")

# Set titles
fig.suptitle(suptitle)
ax1.set_title(title)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save plot
plt.tight_layout(rect=[0, 0.03, 1, 0.95])
plt.savefig("multidiff_71.pdf", bbox_inches="tight")
