
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.gridspec import GridSpec

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# ========== Bar with Error Data (Energy Consumption in Sectors) ==========
# Sectors
sectors = ["Residential", "Commercial", "Industrial", "Transportation"]
# Average Consumption (in TWh)
avg_consumption = [220, 180, 350, 260]
# Error margins
consumption_errors = [10, 12, 15, 9]

# ========== Scatter with Error Data (Energy Provider Efficiency) ==========
# Providers
providers = ["Provider A", "Provider B", "Provider C", "Provider D"]
# Efficiency scores
efficiency_scores = np.random.uniform(80, 95, len(providers))
# Standard deviations
efficiency_stddevs = np.random.uniform(2, 5, len(providers))

# ========== Line with Error Data (Renewable Energy Production) ==========
# Years
years = ["2017", "2018", "2019", "2020"]
# Production Scores
production_regular = np.random.uniform(50, 70, len(years))
production_advanced = np.random.uniform(60, 80, len(years))
# Error margins
production_error = np.random.uniform(2, 3, len(years))

titles = [
    "Energy Consumption in Sectors",
    "Energy Provider Efficiency Scores",
    "Renewable Energy Production Over Years",
]
ylabels = ["Consumption (TWh)", "Efficiency (%)", "Production (TWh)"]
ax2ylim = [75, 100]
ax3labels = ["Regular Methods", "Advanced Methods"]
legend_title = "Production Methods"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and grid layout
fig = plt.figure(figsize=(12, 10))
gs = GridSpec(2, 2, figure=fig)

# Bar with Error Plot
ax1 = fig.add_subplot(gs[0, :])
bar_color = "#ff9999"
ax1.bar(sectors, avg_consumption, yerr=consumption_errors, color=bar_color, capsize=5, ecolor="black")
ax1.set_title(titles[0])
ax1.set_ylabel(ylabels[0])
ax1.grid(True, linestyle='--', alpha=0.7)

# Scatter with Error Plot
ax2 = fig.add_subplot(gs[1, 0])
scatter_color = "#66b3ff"
ax2.errorbar(
    providers,
    efficiency_scores,
    yerr=efficiency_stddevs,
    fmt="D",
    color=scatter_color,
    ecolor="black",
    capsize=5,
    ms=10,
    linestyle='None'
)
ax2.set_title(titles[1])
ax2.set_ylabel(ylabels[1])
ax2.set_ylim(ax2ylim)
ax2.grid(True, linestyle='--', alpha=0.7)

# Line with Error Plot
ax3 = fig.add_subplot(gs[1, 1])
line_color1 = "green"
line_color2 = "blue"
ax3.errorbar(
    years,
    production_regular,
    yerr=production_error,
    fmt="o-",
    color=line_color1,
    ecolor="black",
    capsize=5,
    label=ax3labels[0],
    linestyle='-.'
)
ax3.errorbar(
    years,
    production_advanced,
    yerr=production_error,
    fmt="s-",
    color=line_color2,
    ecolor="black",
    capsize=5,
    label=ax3labels[1],
    linestyle='--'
)
ax3.set_title(titles[2])
ax3.set_ylabel(ylabels[2])
ax3.legend(title=legend_title)
ax3.grid(True, linestyle='--', alpha=0.7)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the figure
plt.tight_layout()
plt.savefig("multidiff_89.pdf", bbox_inches="tight")
