
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde
from matplotlib.gridspec import GridSpec

# ===================
# Part 2: Data Preparation
# ===================
# Seed for reproducibility
import numpy as np

np.random.seed(0)

# Generate synthetic psychology data
# Daily stress levels (scale 1-10)
stress_levels_weekdays = np.random.normal(5, 2, 500)  # Stress during weekdays
stress_levels_weekends = np.random.normal(3, 2, 500)  # Stress during weekends

# Sleep quality scores (scale 1-5) for different groups
sleep_quality_group1 = np.random.normal(3, 0.5, 1000)  # Group 1 sleep quality
sleep_quality_group2 = np.random.normal(4, 0.5, 1000)  # Group 2 sleep quality

# Mood data across a month (days)
days = np.linspace(0, 30, 1000)
mood_variation = 3 + 2 * np.cos(np.linspace(0, 2 * np.pi, 1000))

ax1labels = ["Weekday Stress", "Weekend Stress"]
titles = [
    "Daily Stress Levels",
    "Sleep Quality Scores",
    "Mood Variation Over a Month",
]
xlabels = ["Stress Level (1-10)", "Days"]
ylabels = ["Frequency", "Sleep Quality (1-5)", "Mood (Scale 1-5)"]
ax2xtickslabels = ["Group 1", "Group 2"]
ax2xticks = [1, 2]
bins = np.linspace(0, 10, 20)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the figure using GridSpec
fig = plt.figure(figsize=(10, 10))
gs = GridSpec(2, 2, figure=fig)

# Histogram plot across top (1,1 and 1,2)
ax1 = fig.add_subplot(gs[0, :])
ax1.hist(
    stress_levels_weekdays,
    bins=bins,
    alpha=0.7,
    label=ax1labels[0],
    color="#7fbc41",
    edgecolor="black",
)
ax1.hist(
    stress_levels_weekends,
    bins=bins,
    alpha=0.7,
    label=ax1labels[1],
    color="#b3de69",
    edgecolor="black",
)
ax1.set_title(titles[0])
ax1.set_xlabel(xlabels[0])
ax1.set_ylabel(ylabels[0])
ax1.legend()

# Violin plot on bottom left (2,1)
ax2 = fig.add_subplot(gs[1, 0])
ax2.violinplot([sleep_quality_group1, sleep_quality_group2], showmeans=False, showmedians=True)
ax2.set_title(titles[1])
ax2.set_ylabel(ylabels[1])
ax2.set_xticks(ax2xticks)
ax2.set_xticklabels(ax2xtickslabels)
ax2.grid(True)

# Fill between plot on bottom right (2,2)
ax3 = fig.add_subplot(gs[1, 1])
ax3.fill_between(days, mood_variation, color="brown", alpha=0.2)
ax3.set_title(titles[2])
ax3.set_xlabel(xlabels[1])
ax3.set_ylabel(ylabels[2])

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for a better visual appearance
plt.tight_layout()
plt.savefig("multidiff_92.pdf", bbox_inches="tight")
