
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
# Data for the plot - communication domain (signal strength analysis)
cities_1 = ["New York", "Los Angeles", "Chicago"]
initial_signal_1 = [80, 75, 85]
external_bias_1 = [-10, -8, -12]
internal_signal_1 = [78, 73, 83]
internal_bias_1 = [2, 1, 3]

cities_2 = ["Houston", "Phoenix", "Philadelphia"]
initial_signal_2 = [70, 65, 72]
external_bias_2 = [-5, -9, -7]
internal_signal_2 = [68, 64, 70]
internal_bias_2 = [4, 2, 3]

# Axes Limits and Labels
xlim_values = [50, 90]
ylabel = "Cities"
xlabel = "Signal Strength (dBm)"
title = "Signal Strength Analysis"
external_label = "External Factors\n(Weather Conditions)"
internal_label = "Internal Factors\n(Infrastructure Quality)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, ax1 = plt.subplots(figsize=(8, 6))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# First subplot (cities_1)
for i, city in enumerate(cities_1):
    # External bias line with arrow and dots
    ax1.annotate(
        "",
        xy=(initial_signal_1[i], i + offset * 3 / 2),
        xytext=(initial_signal_1[i] + external_bias_1[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color="green"),
    )
    ax1.scatter(
        [initial_signal_1[i], initial_signal_1[i] + external_bias_1[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="darkgreen",
        s=10,
    )
    ax1.annotate(
        f"{external_bias_1[i]:.2f}",
        (initial_signal_1[i] + external_bias_1[i], i + offset * 1.75),
        color="green",
        ha="right",
        va="center",
    )

# Second subplot (cities_2)
for i, city in enumerate(cities_2):
    # Internal bias line with arrow and dots
    ax1.annotate(
        "",
        xy=(initial_signal_2[i], len(cities_1) + i + offset * 3 / 2),
        xytext=(
            initial_signal_2[i] + external_bias_2[i],
            len(cities_1) + i + offset * 3 / 2,
        ),
        arrowprops=dict(arrowstyle="<-", color="blue"),
    )
    ax1.scatter(
        [initial_signal_2[i], initial_signal_2[i] + external_bias_2[i]],
        [len(cities_1) + i + offset * 3 / 2, len(cities_1) + i + offset * 3 / 2],
        color="darkblue",
        s=10,
    )
    ax1.annotate(
        f"{external_bias_2[i]:.2f}",
        (initial_signal_2[i] + external_bias_2[i], len(cities_1) + i + offset * 1.75),
        color="blue",
        ha="right",
        va="center",
    )

# set y-axis limits
ax1.set_ylim(0, len(cities_1) + len(cities_2))

# Set x-axis limits uniformly
ax1.set_xlim(xlim_values)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(cities_1) + len(cities_2))])
ax1.set_yticklabels(cities_1 + cities_2)

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(cities_1) + len(cities_2))], minor=True)

ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")

# add x-axis grid lines and set gap is 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(5))

ax1.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
green_arrow = mlines.Line2D(
    [],
    [],
    color="green",
    marker=">",
    linestyle="-",
    markersize=8,
    label=external_label,
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)

blue_arrow = mlines.Line2D(
    [],
    [],
    color="blue",
    marker=">",
    linestyle="-",
    markersize=8,
    label=internal_label,
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)

fig.legend(
    handles=[green_arrow, blue_arrow],
    bbox_to_anchor=(0.5, -0.1),
    loc="upper center",
    ncol=2,
)

# Set labels and title
ax1.set_ylabel(ylabel)
ax1.set_xlabel(xlabel)
plt.title(title)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("quiver_26.pdf", bbox_inches="tight")
