
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Data for the plot
# Updated for a new domain: Psychology (Therapies and Techniques)
therapies = ["CBT", "Mindfulness", "Psychoanalysis"]
effectiveness_scores = [8.5, 7.0, 6.0]
effectiveness_bias = [+1.2, +2.1, +1.8]
usability_scores = [6.5, 8.0, 5.5]
usability_bias = [-1.0, -0.5, -2.0]
therapy_labels = ["Effectiveness Bias", "Usability Bias"]

techniques = ["Exposure Therapy", "Meditation", "Free Association"]
effectiveness_scores_tech = [7.0, 8.5, 6.5]
effectiveness_bias_tech = [+1.5, +1.8, +1.2]
usability_scores_tech = [6.0, 7.5, 5.8]
usability_bias_tech = [-1.5, -0.8, -2.2]
tech_labels = ["Effectiveness Bias", "Usability Bias"]

# Labels for the chart
xlabel = "Scores"
ylabel_therapy = "Therapies"
ylabel_tech = "Techniques"
title_therapy = "Psychology Therapies: Effectiveness and Usability Biases"
title_tech = "Psychology Techniques: Effectiveness and Usability Biases"
supertitle = "Psychology Intervention Performance and Usability Biases"
legendlabels_therapy = ["Effectiveness Bias", "Usability Bias"]
legendlabels_tech = ["Effectiveness Bias", "Usability Bias"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(14, 6))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# First subplot (therapies)
for i, therapy in enumerate(therapies):
    # Effectiveness bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(effectiveness_scores[i], i + offset * 3 / 2),
        xytext=(
            effectiveness_scores[i] + effectiveness_bias[i],
            i + offset * 3 / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="solid", color="green"),
    )
    ax1.scatter(
        [
            effectiveness_scores[i],
            effectiveness_scores[i] + effectiveness_bias[i],
        ],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{effectiveness_bias[i]:.2f}",
        (effectiveness_scores[i] + effectiveness_bias[i], i + offset * 1.75),
        color="green",
        ha="left",
        va="center",
    )

    # Usability bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(usability_scores[i], i + offset / 2),
        xytext=(usability_scores[i] + usability_bias[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<|-", linestyle="solid", color="blue"),
    )
    ax1.scatter(
        [
            usability_scores[i],
            usability_scores[i] + usability_bias[i],
        ],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{usability_bias[i]:.2f}",
        (usability_scores[i] + usability_bias[i], i + offset * 0.75),
        color="blue",
        ha="right",
        va="center",
    )

# Second subplot (techniques)
for i, technique in enumerate(techniques):
    ax2.annotate(
        "",
        xy=(effectiveness_scores_tech[i], i + offset * 3 / 2),
        xytext=(
            effectiveness_scores_tech[i] + effectiveness_bias_tech[i],
            i + offset * 3 / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="solid", color="green"),
    )
    ax2.scatter(
        [
            effectiveness_scores_tech[i],
            effectiveness_scores_tech[i] + effectiveness_bias_tech[i],
        ],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{effectiveness_bias_tech[i]:.2f}",
        (effectiveness_scores_tech[i] + effectiveness_bias_tech[i], i + offset * 1.75),
        color="green",
        ha="left",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(usability_scores_tech[i], i + offset / 2),
        xytext=(usability_scores_tech[i] + usability_bias_tech[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<|-", linestyle="solid", color="blue"),
    )
    ax2.scatter(
        [usability_scores_tech[i], usability_scores_tech[i] + usability_bias_tech[i]],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{usability_bias_tech[i]:.2f}",
        (usability_scores_tech[i] + usability_bias_tech[i], i + offset * 0.75),
        color="blue",
        ha="right",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(therapies))
ax2.set_ylim(0, len(techniques))

# Set x-axis limits uniformly
ax1.set_xlim(0, 12)
ax2.set_xlim(0, 12)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(therapies))])
ax1.set_yticklabels(therapies)
ax2.set_yticks([i + offset for i in range(len(techniques))])
ax2.set_yticklabels(techniques)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(therapies))], minor=True)
ax2.set_yticks([i for i in range(len(techniques))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="gray")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="gray")

# Add x-axis grid lines and set gap to 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))
ax2.xaxis.set_major_locator(plt.MultipleLocator(1))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
green_arrow = mlines.Line2D(
    [],
    [],
    color="green",
    marker=">",
    linestyle="solid",
    markersize=8,
    label=legendlabels_therapy[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
blue_arrow = mlines.Line2D(
    [],
    [],
    color="blue",
    marker=">",
    linestyle="solid",
    markersize=8,
    label=legendlabels_therapy[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[green_arrow, blue_arrow], bbox_to_anchor=(0.45, 0.0), ncol=2)

green_arrow = mlines.Line2D(
    [],
    [],
    color="green",
    marker=">",
    linestyle="solid",
    markersize=8,
    label=legendlabels_tech[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
blue_arrow = mlines.Line2D(
    [],
    [],
    color="blue",
    marker=">",
    linestyle="solid",
    markersize=8,
    label=legendlabels_tech[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[green_arrow, blue_arrow], bbox_to_anchor=(0.85, 0.0), ncol=2)

# Add titles and labels
ax1.set_xlabel(xlabel)
ax2.set_xlabel(xlabel)
ax1.set_ylabel(ylabel_therapy)
ax2.set_ylabel(ylabel_tech)
ax1.set_title(title_therapy)
ax2.set_title(title_tech)
plt.suptitle(supertitle)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("quiver_42.pdf", bbox_inches="tight")
