
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Data for the plot
# New domain: Energy Efficiency and Environmental Impact
power_generation_methods = ["Solar", "Wind", "Coal"]
efficiency_scores = [9.5, 8.5, 5.0]
efficiency_bias = [+1.0, +1.5, +2.5]
environmental_impact_scores = [3.0, 2.5, 8.0]
environmental_impact_bias = [-1.0, -0.5, +1.5]
power_labels = ["Efficiency Bias", "Environmental Impact Bias"]

technologies = ["Solar Panel Installation", "Wind Turbine Technology", "Coal Plant Upgrades"]
tech_efficiency_scores = [8.5, 7.5, 6.0]
tech_efficiency_bias = [+1.0, +1.5, +2.0]
tech_environmental_impact_scores = [3.5, 3.0, 8.5]
tech_environmental_impact_bias = [-1.5, -1.0, +1.0]
tech_labels = ["Tech Efficiency Bias", "Environmental Impact Bias"]

# Axis Labels and Other Texts
xlabel = "Score"
ylabel_power = "Power Generation Methods"
ylabel_tech = "Technologies"
title_power = "Power Generation Efficiency and Environmental Impact"
title_tech = "Technology Efficiency and Environmental Impact"
suptitle = "Energy Performance and Environmental Impact"
legend_labels = power_labels  # Reuse the same labels for both charts

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 6))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# First subplot (power generation methods)
for i, method in enumerate(power_generation_methods):
    # Efficiency bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(efficiency_scores[i], i + offset * 3 / 2),
        xytext=(
            efficiency_scores[i] + efficiency_bias[i],
            i + offset * 3 / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="dotted", color="green"),
    )
    ax1.scatter(
        [
            efficiency_scores[i],
            efficiency_scores[i] + efficiency_bias[i],
        ],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{efficiency_bias[i]:.2f}",
        (efficiency_scores[i] + efficiency_bias[i], i + offset * 1.75),
        color="green",
        ha="left",
        va="center",
    )

    # Environmental impact bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(environmental_impact_scores[i], i + offset / 2),
        xytext=(
            environmental_impact_scores[i] + environmental_impact_bias[i],
            i + offset / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="dashed", color="blue"),
    )
    ax1.scatter(
        [
            environmental_impact_scores[i],
            environmental_impact_scores[i] + environmental_impact_bias[i],
        ],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{environmental_impact_bias[i]:.2f}",
        (environmental_impact_scores[i] + environmental_impact_bias[i], i + offset * 0.75),
        color="blue",
        ha="right",
        va="center",
    )

# Second subplot (technologies)
for i, tech in enumerate(technologies):
    ax2.annotate(
        "",
        xy=(tech_efficiency_scores[i], i + offset * 3 / 2),
        xytext=(
            tech_efficiency_scores[i] + tech_efficiency_bias[i],
            i + offset * 3 / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="dotted", color="green"),
    )
    ax2.scatter(
        [
            tech_efficiency_scores[i],
            tech_efficiency_scores[i] + tech_efficiency_bias[i],
        ],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{tech_efficiency_bias[i]:.2f}",
        (tech_efficiency_scores[i] + tech_efficiency_bias[i], i + offset * 1.75),
        color="green",
        ha="left",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(tech_environmental_impact_scores[i], i + offset / 2),
        xytext=(
            tech_environmental_impact_scores[i] + tech_environmental_impact_bias[i],
            i + offset / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="dashed", color="blue"),
    )
    ax2.scatter(
        [
            tech_environmental_impact_scores[i],
            tech_environmental_impact_scores[i] + tech_environmental_impact_bias[i],
        ],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{tech_environmental_impact_bias[i]:.2f}",
        (
            tech_environmental_impact_scores[i] + tech_environmental_impact_bias[i],
            i + offset * 0.75,
        ),
        color="blue",
        ha="right",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(power_generation_methods))
ax2.set_ylim(0, len(technologies))

# Set x-axis limits uniformly
ax1.set_xlim(0, 12)
ax2.set_xlim(0, 12)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(power_generation_methods))])
ax1.set_yticklabels(power_generation_methods)
ax2.set_yticks([i + offset for i in range(len(technologies))])
ax2.set_yticklabels(technologies)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(power_generation_methods))], minor=True)
ax2.set_yticks([i for i in range(len(technologies))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="gray")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="gray")

# Add x-axis grid lines and set gap to 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))
ax2.xaxis.set_major_locator(plt.MultipleLocator(1))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
green_arrow = mlines.Line2D(
    [],
    [],
    color="green",
    marker=">",
    linestyle="dotted",
    markersize=8,
    label=legend_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
blue_arrow = mlines.Line2D(
    [],
    [],
    color="blue",
    marker=">",
    linestyle="dashed",
    markersize=8,
    label=legend_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(
    handles=[green_arrow, blue_arrow],
    bbox_to_anchor=(0.5, 0),
    ncol=2,
    loc="upper center",
)

# Titles and labels
ax1.set_title(title_power)
ax2.set_title(title_tech)
ax1.set_xlabel(xlabel)
ax2.set_xlabel(xlabel)
ax1.set_ylabel(ylabel_power)
ax2.set_ylabel(ylabel_tech)
fig.suptitle(suptitle)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout(rect=[0, 0, 1, 0.95])
plt.savefig("quiver_43.pdf", bbox_inches="tight")
