
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
# Data for the plot
regions_1 = ["North America", "Europe", "Australia"]
forest_area_1 = [7.7, 1.0, 1.2]  # Forest Area (Million sq km)
forest_area_change_1 = [-0.2, -0.05, -0.15]  # Change in Forest Area (Million sq km)
biodiversity_loss_rate_1 = [0.8, 0.5, 0.6]  # Species Extinction Rate (%)
biodiversity_loss_rate_change_1 = [0.05, 0.02, 0.03]  # Change in Extinction Rate (%)

regions_2 = ["South Asia", "Sub-Saharan Africa", "Latin America"]
forest_area_2 = [2.8, 1.5, 8.0]  # Forest Area (Million sq km)
forest_area_change_2 = [-0.7, -0.3, -1.2]  # Change in Forest Area (Million sq km)
biodiversity_loss_rate_2 = [1.2, 1.7, 1.5]  # Species Extinction Rate (%)
biodiversity_loss_rate_change_2 = [0.1, 0.15, 0.2]  # Change in Extinction Rate (%)

ax1_labels = ["Change in Forest Area (Million sq km)", "Change in Extinction Rate (%)"]
ax2_labels = ["Change in Forest Area (Million sq km)", "Change in Extinction Rate (%)"]

xlabel = "Value"
ylabel = "Regions"
title1 = "Developed Regions: Deforestation and Biodiversity Loss"
title2 = "Developing Regions: Deforestation and Biodiversity Loss"
suptitle = "Deforestation and Biodiversity Loss Visualization"
legend_label_forest = "Forest Area"
legend_label_biodiversity = "Species Extinction Rate"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(14, 7))
fig.suptitle(suptitle, fontsize=16)

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# Function to plot on the given axis
def plot_climate_data(
    ax,
    regions,
    forest_area,
    forest_area_change,
    biodiversity_loss_rate,
    biodiversity_loss_rate_change,
    labels,
    title,
):
    for i, region in enumerate(regions):
        # Forest Area change line with arrow and grey dots at start and end
        ax.annotate(
            "",
            xy=(forest_area[i], i + offset * 3 / 2),
            xytext=(
                forest_area[i] + forest_area_change[i],
                i + offset * 3 / 2,
            ),
            arrowprops=dict(arrowstyle="<-", color="grey"),
        )
        ax.scatter(
            [
                forest_area[i],
                forest_area[i] + forest_area_change[i],
            ],
            [i + offset * 3 / 2, i + offset * 3 / 2],
            color="black",
            s=25,
        )
        ax.annotate(
            f"{forest_area_change[i]:.2f}",
            (forest_area[i] + forest_area_change[i], i + offset * 1.75),
            color="grey",
            ha="right",
            va="center",
        )

        # Biodiversity Loss Rate change line with arrow and green dots at start and end
        ax.annotate(
            "",
            xy=(biodiversity_loss_rate[i], i + offset / 2),
            xytext=(biodiversity_loss_rate[i] + biodiversity_loss_rate_change[i], i + offset / 2),
            arrowprops=dict(arrowstyle="<-", color="green"),
        )
        ax.scatter(
            [biodiversity_loss_rate[i], biodiversity_loss_rate[i] + biodiversity_loss_rate_change[i]],
            [i + offset / 2, i + offset / 2],
            color="black",
            s=25,
        )
        ax.annotate(
            f"{biodiversity_loss_rate_change[i]:.2f}",
            (biodiversity_loss_rate[i] + biodiversity_loss_rate_change[i], i + offset * 0.75),
            color="green",
            ha="left",
            va="center",
        )

    # set axis limits
    ax.set_ylim(0, len(regions))
    ax.set_xlim(-2, 10)
    # Set y-ticks and labels
    ax.set_yticks([i + offset for i in range(len(regions))])
    ax.set_yticklabels(regions)
    ax.set_xlabel(xlabel)
    ax.set_ylabel(ylabel)
    ax.set_title(title, fontsize=14)

    # Offset grid lines on the y-axis
    ax.set_yticks([i for i in range(len(regions))], minor=True)
    ax.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="grey")

    # add x-axis grid lines
    ax.xaxis.set_major_locator(plt.MultipleLocator(2))
    ax.grid(axis="x", linestyle="--", linewidth=0.5)


# Plot data
plot_climate_data(
    ax1,
    regions_1,
    forest_area_1,
    forest_area_change_1,
    biodiversity_loss_rate_1,
    biodiversity_loss_rate_change_1,
    ax1_labels,
    title1,
)
plot_climate_data(
    ax2,
    regions_2,
    forest_area_2,
    forest_area_change_2,
    biodiversity_loss_rate_2,
    biodiversity_loss_rate_change_2,
    ax2_labels,
    title2,
)

# Set legends
grey_arrow = mlines.Line2D(
    [],
    [],
    color="grey",
    marker=">",
    linestyle="-",
    markersize=8,
    label=legend_label_forest,
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
green_arrow = mlines.Line2D(
    [],
    [],
    color="green",
    marker=">",
    linestyle="-",
    markersize=8,
    label=legend_label_biodiversity,
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(
    handles=[grey_arrow, green_arrow],
    bbox_to_anchor=(0.5, -0.05),
    loc="upper center",
    ncol=2,
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout(rect=[0, 0.03, 1, 0.95])
plt.savefig("quiver_45.pdf", bbox_inches="tight")
