
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Data for the plot
continents_1 = ["Asia", "Africa", "Europe"]
population_1 = [4500, 1300, 750]  # Population in millions
population_change_1 = [150, 50, 5]  # Change in population in millions
urban_population_1 = [2500, 550, 600]  # Urban population in millions
urban_population_change_1 = [100, 30, 10]  # Change in urban population in millions

continents_2 = ["North America", "South America", "Australia/Oceania"]
population_2 = [580, 430, 42]  # Population in millions
population_change_2 = [10, 20, 1]  # Change in population in millions
urban_population_2 = [450, 350, 30]  # Urban population in millions
urban_population_change_2 = [5, 10, 0.5]  # Change in urban population in millions

# Axis labels and legend labels
ax1_labels = ["Population Change (millions)", "Urban Population Change (millions)"]
ax2_labels = ["Population Change (millions)", "Urban Population Change (millions)"]

suptitle = "Historical Population Trends by Continent"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(14, 7))
offset = 0.5

# Color variables
pop_change_color = "green"
urban_pop_change_color = "purple"

# First subplot (continents_1)
for i, continent in enumerate(continents_1):
    # Population change line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(population_1[i], i + offset * 3 / 2),
        xytext=(population_1[i] + population_change_1[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<|-", color=pop_change_color),
    )
    ax1.scatter(
        [population_1[i], population_1[i] + population_change_1[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color=pop_change_color,
        s=50,
        edgecolor="black",
    )
    ax1.annotate(
        f"{population_change_1[i]:.2f}",
        (population_1[i] + population_change_1[i], i + offset * 1.75),
        color=pop_change_color,
        ha="right",
        va="center",
    )

    # Urban Population change line with arrow and dots at start and end
    ax1.annotate(
        "",
        xy=(urban_population_1[i], i + offset / 2),
        xytext=(
            urban_population_1[i] + urban_population_change_1[i],
            i + offset / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", color=urban_pop_change_color),
    )
    ax1.scatter(
        [
            urban_population_1[i],
            urban_population_1[i] + urban_population_change_1[i],
        ],
        [i + offset / 2, i + offset / 2],
        color=urban_pop_change_color,
        s=50,
        edgecolor="black",
    )
    ax1.annotate(
        f"{urban_population_change_1[i]:.2f}",
        (
            urban_population_1[i] + urban_population_change_1[i],
            i + offset * 0.75,
        ),
        color=urban_pop_change_color,
        ha="left",
        va="center",
    )

# Second subplot (continents_2)
for i, continent in enumerate(continents_2):
    ax2.annotate(
        "",
        xy=(population_2[i], i + offset * 3 / 2),
        xytext=(population_2[i] + population_change_2[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<|-", color=pop_change_color),
    )
    ax2.scatter(
        [population_2[i], population_2[i] + population_change_2[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color=pop_change_color,
        s=50,
        edgecolor="black",
    )
    ax2.annotate(
        f"{population_change_2[i]:.2f}",
        (population_2[i] + population_change_2[i], i + offset * 1.75),
        color=pop_change_color,
        ha="right",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(urban_population_2[i], i + offset / 2),
        xytext=(
            urban_population_2[i] + urban_population_change_2[i],
            i + offset / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", color=urban_pop_change_color),
    )
    ax2.scatter(
        [
            urban_population_2[i],
            urban_population_2[i] + urban_population_change_2[i],
        ],
        [i + offset / 2, i + offset / 2],
        color=urban_pop_change_color,
        s=50,
        edgecolor="black",
    )
    ax2.annotate(
        f"{urban_population_change_2[i]:.2f}",
        (
            urban_population_2[i] + urban_population_change_2[i],
            i + offset * 0.75,
        ),
        color=urban_pop_change_color,
        ha="left",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(continents_1))
ax2.set_ylim(0, len(continents_2))

# Set x-axis limits uniformly
ax1.set_xlim(0, 5000)
ax2.set_xlim(0, 1000)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(continents_1))])
ax1.set_yticklabels(continents_1)
ax2.set_yticks([i + offset for i in range(len(continents_2))])
ax2.set_yticklabels(continents_2)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(continents_1))], minor=True)
ax2.set_yticks([i for i in range(len(continents_2))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="grey")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="grey")

# Add x-axis grid lines
ax1.xaxis.set_major_locator(plt.MultipleLocator(500))
ax2.xaxis.set_major_locator(plt.MultipleLocator(100))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Create legend entries
green_arrow = mlines.Line2D(
    [],
    [],
    color=pop_change_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax1_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
purple_arrow = mlines.Line2D(
    [],
    [],
    color=urban_pop_change_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax1_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(
    handles=[green_arrow, purple_arrow],
    bbox_to_anchor=(0.70, 0.01),
    ncol=2,
    frameon=False,
)

# plt.subplots_adjust(bottom=0.15)
plt.suptitle(suptitle, fontsize=16)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("quiver_47.pdf", bbox_inches="tight")
