# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines
import matplotlib.patches as mpatches

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Define the vector field function for wind patterns
def wind_vector_field(X, Y):
    # Placeholder function for the vector field
    U = np.sin(X)
    V = np.cos(Y)
    return U, V

def modified_wind_vector_field(X, Y):
    # Modified wind patterns
    U = np.sin(X) + np.cos(Y)
    V = np.cos(X) - np.sin(Y)
    return U, V

# Create a grid of points representing weather scenarios
x = np.linspace(-np.pi, np.pi, 30)
y = np.linspace(-np.pi, np.pi, 30)
X, Y = np.meshgrid(x, y)

# Compute the vector field
U, V = wind_vector_field(X, Y)

# Compute the modified vector field
U_mod, V_mod = modified_wind_vector_field(X, Y)

# Plot the curves representing different temperature trends
x = np.linspace(-np.pi/2, np.pi/2, 100)  # Updated range for temperature scenarios
xlabel = "Latitude (degrees)"
ylabel = "Wind Speed (m/s)"
patch_labels = ["Wind Pattern 1", "Wind Pattern 2"]
line_labels = ["Temperature Trend 1", "Temperature Trend 2", "Adjusted Temperature 1", "Adjusted Temperature 2"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(8, 6))

# Colors for the plots
supply_influence_color = "#1f77b4"
demand_influence_color = "#ff7f0e"
supply_curve_color = "#2ca02c"
demand_curve_color = "#d62728"
adjusted_supply_color = "#9467bd"
adjusted_demand_color = "#8c564b"

# Quiver plots for wind patterns
plt.quiver(X, Y, U, V, color=supply_influence_color, alpha=0.6)  # Wind Pattern 1
plt.quiver(X, Y, U_mod, V_mod, color=demand_influence_color, alpha=0.6)  # Wind Pattern 2

# Plot different temperature trends
plt.plot(x, 1.2 * np.sin(x), color=supply_curve_color, linestyle="-", linewidth=2)
plt.plot(x, 0.8 * np.sin(x) + 1.5, color=demand_curve_color, linestyle="--", linewidth=2)
plt.plot(x, 0.5 * np.sin(x) - 1, color=adjusted_supply_color, linestyle="-.", linewidth=2)
plt.plot(x, 0.3 * np.sin(x) + 0.5, color=adjusted_demand_color, linestyle=":", linewidth=2)

# Add labels and legend
plt.xlabel(xlabel, fontsize=14, style="italic")
plt.ylabel(ylabel, fontsize=14, style="italic")

# Patches for vector fields
blue_patch = mpatches.Patch(color=supply_influence_color, label=patch_labels[0], alpha=0.6)
orange_patch = mpatches.Patch(color=demand_influence_color, label=patch_labels[1], alpha=0.6)

# Lines for temperature trends
supply_curve = mlines.Line2D(
    [], [], color=supply_curve_color, linestyle="-", label=line_labels[0], linewidth=2
)
demand_curve = mlines.Line2D(
    [], [], color=demand_curve_color, linestyle="--", label=line_labels[1], linewidth=2
)
adjusted_supply = mlines.Line2D(
    [], [], color=adjusted_supply_color, linestyle="-.", label=line_labels[2], linewidth=2
)
adjusted_demand = mlines.Line2D(
    [], [], color=adjusted_demand_color, linestyle=":", label=line_labels[3], linewidth=2
)

# Combine all legend handles
handles = [
    blue_patch,
    orange_patch,
    supply_curve,
    demand_curve,
    adjusted_supply,
    adjusted_demand,
]

# Add the legend to the plot with specified location
plt.legend(handles=handles, loc="upper right")

# Set y-axis limit to -3 to 3
plt.ylim(-3, 3)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("quiver_51.pdf", bbox_inches="tight")
